/* $Id: MsqlLog.java,v 2.1 1999/03/03 03:27:10 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import java.io.PrintWriter;
import java.sql.DriverManager;

/**
 * Provides logging for objects in the mSQL-JDBC driver. Each object in
 * the driver has an <CODE>MsqlLog</CODE> object that supports the
 * logging of certain kinds of events as specified at runtime by
 * either system properties or connection properties. You can associate
 * logging with either the whole application or a specific connection.
 * If you specify both, the connection properties will override
 * the system properties for that connection only.
 * <P>
 * The system properties have the same names as the connection level
 * properties, except they are preceeded by &quot;imaginary.msql-jdbc.&quot;.
 * So, if you want to log regular errors and fatal errors, you would pass:
 * <PRE>
 * -Dimaginary.msql-jdbc.logging=FATAL,ERROR
 * </PRE>
 * If you want to set that logging just for a specific connection,
 * then do the following in your connection code:
 * <PRE>
 * Properties p = new Properties();
 * // set all the things like user name and password the driver requires
 * p.setProperty("logging", "FATAL,ERROR");
 * </PRE>
 * The different kinds of levels are:
 * <DL>
 * <DT> FATAL
 * <DD> Fatal errors.
 * <DT> JDBC
 * <DD> JDBC events.
 * <DT> MSQL
 * <DD> mSQL events.
 * <DT> ERROR
 * <DD> Non-fatal errors.
 * <DT> DRIVER
 * <DD> Internal events for this driver... this is really noisy.
 * <DT> METHODS
 * <DD> Method names are logged with every event.
 * <DT> THREADS
 * <DD> Thread names are logged with every event.
 * <DT> ALL
 * <DD> Everything.
 * </DL>
 */
public class MsqlLog {
    static public final int NONE    = 0;
    static public final int FATAL   = 1;
    static public final int JDBC    = 2;
    static public final int MSQL    = 4;
    static public final int ERROR   = 8;
    static public final int DRIVER  = 16;
    static public final int METHODS = 32;
    static public final int THREADS = 64;
    static public final int ALL     = (FATAL|JDBC|MSQL|ERROR|DRIVER|METHODS|THREADS);
    
    private int    level = MsqlLog.NONE;
    private Object owner = null;

    /**
     * Constructs a new logger that logs things foor the specified levels
     * for the specified object.
     * @param lvl a bit map of logging levels containing the type of
     * events that should be logged
     * @param own the object for whom logging is being performed
     */
    public MsqlLog(int lvl, Object own) {
	super();
	level = lvl;
	owner = own;
    }

    /**
     * Closes out the log by flushing any changes.
     */
    public void close() {
	PrintWriter pw = DriverManager.getLogWriter();

	if( pw != null ) {
	    pw.flush();
	}
    }

    /**
     * @return the level of events being logged
     */
    public int getLevel() {
	return level;
    }
    
    /**
     * Attempts to log an event of a specified type to the driver
     * manager's print writer or to stderr/stdout if no print writer is
     * set.
     * @param method the method from which logging is being done
     * @param type the type event being logged
     * @param event the event being logged
     */
    public void log(String method, int type, String event) {
	PrintWriter log;
	String str;

	if( (level & type) == 0 ) {
	    return;
	}
	switch( type ) {
	case MsqlLog.NONE:
	    return;
	case MsqlLog.FATAL:
	    str = "FATAL ERROR";
	    break;
	case MsqlLog.JDBC:
	    str = "JDBC";
	    break;
	case MsqlLog.MSQL:
	    str = "mSQL";
	    break;
	case MsqlLog.ERROR:
	    str = "ERROR";
	    break;
	case MsqlLog.DRIVER:
	    str = "DRIVER";
	    break;
	default:
	    str = "UNK";
	    break;
	}
	if( (level & MsqlLog.METHODS) != 0 ) {
	    str = str + " " + owner.getClass().getName() + "." + method;
	}
	else {
	    str = str;
	}
	if( (level & MsqlLog.THREADS) != 0 ) {
	    str = str + " (" + Thread.currentThread().getName() + ")";
	}
	str = str + ": " + event;
	log = DriverManager.getLogWriter();
	if( log == null ) {
	    if( type == MsqlLog.ERROR || type == MsqlLog.FATAL ) {
		System.err.println(str);
	    }
	    else {
		System.out.println(str);
	    }
	}
	else {
	    log.println(str);
	}
    }
}
