/* $Id: MsqlDataSource.java,v 2.3 1999/01/21 01:48:15 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import java.io.PrintWriter;
import java.io.Serializable;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Properties;
import javax.naming.NamingException;
import javax.naming.Reference;
import javax.naming.Referenceable;
import javax.naming.StringRefAddr;
import javax.sql.DataSource;

/**
 * JDBC 2.0 standard extension data source for JNDI access to an mSQL
 * data source.  Given a JNDI-accessible naming and directory service
 * and a registered mSQL data source, client code can access that
 * data source using JNDI calls.  To access a database called
 * &quot;Wibbles&quot;, you would use the following code:
 * <PRE>
 * Context ctx = new InitialContext();
 * DataSource ds = (DataSource)ctx.lookup("jdbc/Wibbles");
 * Connection conn = ds.getConnection("borg", "");
 * </PRE>
 * You must specify the following attributes in your JNDI directory:
 * <DL>
 * <DT> <CODE>databaseName</CODE>
 * <DD> The name of the mSQL database.
 * <DT> <CODE>encoding</CODE>
 * <DD> The character set encoding for this connection.  If you don't know
 * what that means, set it to "8859_1" and everything will work fine.
 * <DT> <CODE>loginTimeout</CODE>
 * <DD> This is not used yet, so set it to 0.
 * <DT> <CODE>password</CODE>
 * <DD> You can configure a data source with a password to use for
 * connections, thereby not requiring individual applications to know
 * a user name/password.  Right now, this does not matter for mSQL
 * since mSQL does not require passwords.  But it might in the future.
 * <DT> <CODE>port</CODE>
 * <DD> The server port to which mSQL is listening.  If you do not know
 * this value, try 1114.
 * <DT> <CODE>serverName</CODE>
 * <DD> The name of the server on which mSQL is running.
 * <DT> <CODE>user</CODE>
 * <DD> The user name to use for the database connection.
 * </DL>
 * <BR>
 * Last modified $Date: 1999/01/21 01:48:15 $
 * @version $Revision: 2.3 $
 * @author George Reese (borg@imaginary.com)
 */
public class MsqlDataSource
implements DataSource, Referenceable, Serializable {
    // The name of the mSQL database
    private           String      databaseName = null;
    // The encoding to use, 8859_1 by default
    private           String      encoding     = "8859_1";
    // A log stream
    private transient PrintWriter logWriter    = null;
    // The login timeout, currently unused
    private           int         loginTimeout = 0;
    // The default password to use for connections
    private           String      password     = "";
    // The port on which mSQL is listening, 1114 by default
    private           int         port         = 1114;
    // The name of the server on which mSQL is listening
    private           String      serverName   = null;
    // The default user to use for connections
    private           String      user         = "";

    /**
     * Constructs a new <CODE>MsqlDataSource</CODE> without any values.  It
     * is also used by serialization.
     */
    public MsqlDataSource() {
	super();
    }

    /**
     * Creates an mSQL connection using the configured user name and
     * password for the JNDI directory entry.
     * @return a database connection
     * @throws java.sql.SQLException the data source was unable to create
     * a database connection using the configured user and password
     */
    public Connection getConnection() throws SQLException {
	return getConnection(user, password);
    }

    /**
     * Provides a database connection using a user and password not
     * configured in the JNDI directory.
     * @param uid the user ID to connect with
     * @param pw the password to use for the connection
     * @return a database connection
     * @throws java.sql.SQLException the data source was unable to create
     * a database connection using the specified user and password
     */
    public Connection getConnection(String uid, String pw)
    throws SQLException {
	Properties props = new Properties();

	if( uid == null ) {
	    if( user == null ) {
		uid = "";
	    }
	    else {
		uid = user;
	    }
	}
	if( pw == null ) {
	    if( password == null ) {
		password = "";
	    }
	    else {
		password = pw;
	    }
	}
	props.put("user", uid);
	props.put("password", pw);
	props.put("encoding", encoding);
	return getConnection(props);
    }

    /**
     * Creates a connection using the specified properties.
     * @param props the properties to use for the connection
     * @return a database connection
     * @throws java.sql.SQLException a database error occurred
     */
    protected Connection getConnection(Properties props)
    throws SQLException {
	return new MsqlConnection(getURL(), getServerName(), getPort(),
				  getDatabaseName(), props);
    }

    /**
     * @return the name of the mSQL database to connect to
     */
    public String getDatabaseName() {
	return databaseName;
    }

    /**
     * @return the character set encoding
     */
    public String getEncoding() {
	return encoding;
    }
    
    /**
     * @return the error log.
     * @throws java.sql.SQLException could not get a stream
     */
    public PrintWriter getLogWriter() throws SQLException {
	return logWriter;
    }

    /**
     * This feature is not implemented.
     * @return 0
     * @throws java.sql.SQLException this is never thrown
     */
    public int getLoginTimeout() throws SQLException {
	return loginTimeout;
    }

    /**
     * @return the mSQL port on which the database is listening.
     */
    public int getPort() {
	return port;
    }

    /**
     * Required method to support this class as a <CODE>Referenceable</CODE>.
     * @return a reference to this object
     * @throws javax.naming.NamingException a problem occurred constructing
     * the reference
     */
    public Reference getReference() throws NamingException {
	String cname = "com.imaginary.sql.msql.MsqlDataSourceFactory";
	Reference ref = new Reference(getClass().getName(), cname, null);

	ref.add(new StringRefAddr("serverName", getServerName()));
	ref.add(new StringRefAddr("databaseName", getDatabaseName()));
	ref.add(new StringRefAddr("port", "" + getPort()));
	ref.add(new StringRefAddr("encoding", getEncoding()));
	ref.add(new StringRefAddr("user", getUser()));
	ref.add(new StringRefAddr("password", password));
	return ref;
    }

    /**
     * @return the name of the server on which mSQL resides
     */
    public String getServerName() {
	return serverName;
    }

    /**
     * The JDBC URL that would be used to create the same sort of
     * connection used by this data source.
     * @return the JDBC URL for this data source
     */
    public String getURL() {
	String url = "jdbc:msql://";

	url = url + getServerName() + ":" + getPort()+ "/"+ getDatabaseName();
	return url;
    }

    /**
     * @return user ID for the connection
     */
    public String getUser() {
	return user;
    }
    
    /**
     * Sets the database name.
     * @param nom the name of the database
     */
    public void setDatabaseName(String nom) {
	databaseName = nom;
    }

    /**
     * Sets the encoding to use for this connection.
     * @param enc the encoding
     */
    public void setEncoding(String enc) {
	encoding = enc;
    }
    
    /**
     * Sets the log writer for this data source.
     * @param out the new log writer
     * @throws java.sql.SQLException this is never thrown
     */
    public void setLogWriter(PrintWriter out) throws SQLException {
	logWriter = out;
    }

    /**
     * This feature is not yet implemented.
     * @param seconds ignore this
     * @return java.sql.SQLException this is never thrown
     */
    public void setLoginTimeout(int seconds) throws SQLException {
    }

    /**
     * Sets the password to use for connecting to the database
     * @param pass the password
     */
    public void setPassword(String pass) {
	password = pass;
    }

    /**
     * Sets the database port.
     * @param p the port
     */
    public void setPort(int p) {
	port = p;
    }

    /**
     * Sets the server name.
     * @param nom the server name
     */
    public void setServerName(String nom) {
	serverName = nom;
    }

    /**
     * Sets the user ID.
     * @param uid the user ID
     */
    public void setUser(String uid) {
	user = uid;
    }
}
