/* $Id: MsqlConnection.java,v 2.4 1999/07/06 05:50:53 borg Exp $ */
/* Copyright (c) 1997-1998 George Reese */
package com.imaginary.sql.msql;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.ref.WeakReference;
import java.net.Socket;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

/**
 * The MsqlConnection class is an implementation of the JDBC Connection
 * interface that represents a database transaction.  This class should
 * never be directly instantiated by an application, but instead should
 * be gotten by calling DriverManager.getConnection().
 * <BR>
 * Last modified $Date: 1999/07/06 05:50:53 $
 * @version $Revision: 2.4 $
 * @author George Reese (borg@imaginary.com)
 * @see java.sql.DriverManager#getConnection
 */
public class MsqlConnection implements Connection {
    // This is the network connection to the database
    private   Socket           connection     = null;
    // The database to which this connection is connected
    protected String           database       = null;
    // This is the encoding of the client application character data
    private   String           encoding       = "8859_1";
    // Notes that the connection is currently in use
    private   boolean          inUse          = false;
    // The input stream to the database
    protected MsqlInputStream  input          = null;
    // The logger for this connection
    private   MsqlLog          log            = null;
    // The output stream to the database
    protected MsqlOutputStream output         = null;
    // A list of open statements
    private   ArrayList        statements     = new ArrayList();
    // The type map for this connection
    private   Map              typeMap        = new HashMap();
    // The URL used to connect to the database
    protected String           url            = null;
    // The name of the user connecting to the database
    protected String           user           = null;
    // The major version number of the mSQL server
    protected int              version        = 0;
    // The version string that mSQL provides
    protected String           versionString  = null;

    /**
     * Constructs a new connection to an mSQL database and makes the
     * connection.
     * @param u the JDBC URL for the connection
     * @param host the name of the host on which the database resides
     * @param port the port on which mSQL is listening
     * @param db the name of the database to which to connect
     * @param p the properties to use for making the connection
     * @exception java.sql.SQLException an error occurred making the connection
     */
    MsqlConnection(String u, String host, int port, String db, Properties p)
    throws SQLException {
	super();
	{
	    String ll;

	    if( p.containsKey("logging") ) {
		ll = (String)p.get("logging");
	    }
	    else {
		ll = System.getProperty("imaginary.msql-jdbc.logging", "NONE");
	    }
	    {
		StringTokenizer tokens = new StringTokenizer(ll, ",");
		int lvl = 0;

		while( tokens.hasMoreTokens() ) {
		    String l = tokens.nextToken().trim();

		    if( l.equals("NONE") ) {
			continue;
		    }
		    else if( l.equals("ALL") ) {
			lvl = MsqlLog.ALL;
			break;
		    }
		    else if( l.equals("FATAL") ) {
			lvl |= MsqlLog.FATAL;
		    }
		    else if( l.equals("JDBC") ) {
			lvl |= MsqlLog.JDBC;
		    }
		    else if( l.equals("MSQL") ) {
			lvl |= MsqlLog.MSQL;
		    }
		    else if( l.equals("ERROR") ) {
			lvl |= MsqlLog.ERROR;
		    }
		    else if( l.equals("DRIVER") ) {
			lvl |= MsqlLog.DRIVER;
		    }
		}
		log = new MsqlLog(lvl, this);
		log.log("MsqlConnection()", MsqlLog.DRIVER,
			"Assigned logging level of " + ll);
	    }
	}
	url = u;
	user = p.getProperty("user", "nobody");
	connect(host, port, user);
	setCatalog(db);
	encoding = p.getProperty("encoding");
	if( encoding == null || encoding.equals("") ) {
	    encoding = System.getProperty("imaginary.msql-jdbc.encoding",
					  "8859_1");
	}
	log.log("MsqlConnection()", MsqlLog.DRIVER,
		"Set encoding to " + encoding);
	try {
	    byte[] b = new byte[1];
	    String tmp;

	    b[0] = (byte)'p';
	    tmp = new String(b, encoding);
	}
	catch( UnsupportedEncodingException e ) {
	    log.log("MsqlConnection()", MsqlLog.ERROR,
		    "Unsupported encoding, using 8859_1");
	    encoding = "8859_1";
	}
	Thread t = new Thread() {
	    public void run() {
		while( !isClosed() ) {
		    try { Thread.sleep(300000); }
		    catch( InterruptedException e ) { }
		}
		cleanStatements();
	    }
	};
	t.setPriority(Thread.MIN_PRIORITY);
	t.setDaemon(true);
	log.log("MsqlConnection()", MsqlLog.DRIVER,
		"Starting thread for database connection.");
	t.start();
    }

    /**
     * The connection can be used for only one task at a time since
     * mSQL processing works on a FIFO basis.  Thus each resource
     * belonging to this connection needs to request control of its I/O
     * streams.  This method forms that request.  However, each resource
     * also needs to be good and release it.
     */
    synchronized void capture() {
	log.log("capture()", MsqlLog.DRIVER,
		"Capture attempted by " +
		Thread.currentThread().getName() + ".");
	while( isInUse() ) {
	    try { wait(1500); }
	    catch( InterruptedException e ) { }
	}
	log.log("capture()", MsqlLog.DRIVER,
		"Capture succeeded by " +
		Thread.currentThread().getName() + ".");
	inUse = true;
    }
    
    /**
     * Cleans up this connection's resources.
     */
    private synchronized void clean() {
	Iterator stmts;

	log.log("clean()", MsqlLog.DRIVER,
		"Cleaning up connection resources.");
	stmts = statements.iterator();
	while( stmts.hasNext() ) {
	    WeakReference ref = (WeakReference)stmts.next();
	    Statement stmt = (Statement)ref.get();

	    if( stmt != null ) {
		try {
		    stmt.close();
		}
		catch( SQLException e ) {
		    e.printStackTrace();
		}
	    }
	}
	statements.clear();
	if( input != null ) {
	    try { input.close(); }
	    catch( IOException e ) { }
	    input = null;
	}
	if( output != null ) {
	    try { output.close(); }
	    catch( IOException e ) { }
	    output = null;
	}
	if( connection != null ) {
	    try { connection.close(); }
	    catch( IOException e ) { }
	    connection = null;
	}
	user = null;
	release();
    }

    /**
     * Cleans out old statements.
     */
    private synchronized void cleanStatements() {
	Iterator stmts = statements.iterator();
	int i = 0, cleaned = 0;
	
	while( stmts.hasNext() ) {
	    WeakReference ref = (WeakReference)stmts.next();

	    if( ref.get() == null ) {
		statements.remove(i);
		cleaned++;
	    }
	    i++;
	}
	log.log("cleanStatements()", MsqlLog.DRIVER,
		"Cleaned " + cleaned + " statements.");
    }
    
    /**
     * Since mSQL produces no warnings, this is a NO-OP
     * @exception java.sql.SQLException this is never thrown
     */
    public void clearWarnings() throws SQLException {
	log.log("clearWarnings()", MsqlLog.JDBC, "Warnings cleared.");
    }

    /**
     * Closes this database connection.  It will also close any associated
     * Statement instances.
     * @exception java.sql.SQLException thrown if problems occur
     */
    public synchronized void close() throws SQLException {
	Exception except = null;

	log.log("close()", MsqlLog.JDBC, "Closing connection.");
	if( isClosed() ) {
	    log.log("close()", MsqlLog.ERROR,
		    "Cannot close a closed connection.");
	    throw new MsqlException("This connection is already closed.");
	}
	// make sure no one is using the database connection
	capture();
	// Let mSQL know we are closing down
	log.log("close()", MsqlLog.MSQL, "Sending close to server.");
	try {
	    output.writeString("1", encoding);
	    output.flush();
	}
	catch( IOException e ) {
	    // save this exception to throw after everything is cleaned
	    except = e;
	    log.log("close()", MsqlLog.ERROR,
		    "Failed to send close to server.");
	}
	// close all resources and reset values
	clean();
	log.close();
	if( except != null ) {
	    throw new MsqlException(except);
	}
    }

    /**
     * This is a NO-OP for mSQL.  All statements are always auto-committed.
     * @exception java.sql.SQLException this is never thrown
     */
    public void commit() throws SQLException {
	log.log("commit()", MsqlLog.JDBC, "Commit received.");
    }

    /**
     * Connects to the database and initialized the IO streams that are
     * used by the rest of the driver.
     * @param host the server on which the database sits
     * @param port the port to which mSQL is listening
     * @param user the user ID with which to make the connection
     * @exception java.sql.SQLException could not connect to the database
     */
    private synchronized void connect(String host, int port, String user)
    throws SQLException {
	String tmp;
	
	log.log("connect()", MsqlLog.JDBC,
		"Connect to " + host + ":" + port + " under UID " + user);
	capture();
	log.log("connect()", MsqlLog.DRIVER, "Created socket.");
	try {
	    connection = new Socket(host, port);
	    input = new MsqlInputStream(connection.getInputStream());
	    output = new MsqlOutputStream(connection.getOutputStream());
	}
	catch( IOException e ) {
	    log.log("connect()", MsqlLog.FATAL,
		    "Socket and stream creation failed: " + e.getMessage());
	    clean();
	    throw new MsqlException(e);
	}
	try {
	    tmp = input.readString(encoding);
	}
	catch( IOException e ) {
	    log.log("connect()", MsqlLog.FATAL,
		    "Failed to read version response from the server: " +
		    e.getMessage());
	    clean();
	    throw new MsqlException(e);
	}
	// Check for mSQL version
	if( tmp.startsWith("0:22:") || tmp.startsWith("0:23:") ) {
	    log.log("connect()", MsqlLog.MSQL,
		    "Responded with version \"" + tmp + "\"");
	    version = 2; // version 2.0
	    versionString = tmp.substring(5);
	}
	else if( tmp.startsWith("0:6:") ) {
	    log.log("connect()", MsqlLog.MSQL,
		    "Responded with version \"" + tmp + "\"");
	    version = 1; // version 1.0.x
	    versionString = tmp.substring(4);
	}
        else {
	    log.log("connect()", MsqlLog.MSQL,
		    "Responded with version \"" + tmp + "\"");
	    log.log("connect()", MsqlLog.FATAL, "Unsupported verion string.");
	    clean();
	    throw new MsqlException("Unsupported mSQL version.");
	}
        log.log("connect()", MsqlLog.MSQL, "Sending user name");
	try {
	    // Check if user was validated
	    output.writeString(user, encoding);
	}
	catch( IOException e ) {
	    log.log("connect()", MsqlLog.FATAL,
		    "Failed to send user name: " + e.getMessage());
	    clean();
	    throw new MsqlException(e);
	}
	try {
	    tmp = input.readString(encoding);
	}
	catch( IOException e ) {
	    log.log("connect()", MsqlLog.FATAL,
		    "Failed to read user name validation from server: " +
		    e.getMessage());
	    clean();
	    throw new MsqlException(e);
	}	    
	if( !tmp.startsWith("-100:") ) {
	    log.log("connect()", MsqlLog.MSQL,
		    "Responded with \"" + tmp + "\"");
	    log.log("connect()", MsqlLog.FATAL, "Access denied.");
	    clean();
	    throw new MsqlException("Access to server denied.");
	}
	release();
    }

    /**
     * This JDBC method creates an instance of MsqlStatement that is
     * forward-only and read-only.  Note that mSQL does not provide a way to
     * interrupt executing statements, so it has to wait for any
     * pending statements to finish before closing them.  
     * @return a new MsqlStatement instance
     * @exception java.sql.SQLException an error occurred in creating the
     * Statement instance, likely raised by the constructor
     */
    public synchronized Statement createStatement() throws SQLException {
	return createStatement(ResultSet.TYPE_FORWARD_ONLY,
			       ResultSet.CONCUR_READ_ONLY);
    }

    /**
     * Creates a JDBC Statement whose result sets have the specified
     * cursor type and concurrency model.
     * @param type the result set type for the statement
     * @param concur the result set concurrency
     * @return a brand new Statement
     * @exception java.sql.SQLException the Connection was unable to create a
     * new Statement
     */
    public synchronized Statement createStatement(int type, int concur)
    throws SQLException {
	WeakReference ref;
	Statement stmt;

	log.log("createStatement()", MsqlLog.JDBC,
		"Creating statement (type=" + type + ",concur=" +
		concur + ").");
	stmt = new MsqlStatement(this, type, concur, log.getLevel());
	ref = new WeakReference(stmt);
	statements.add(ref);
	return stmt;
    }

    /**
     * This method always returns true since mSQL is always in auto-commit
     * mode.
     * @return true--always
     * @exception java.sql.SQLException never thrown
     */
    public boolean getAutoCommit() throws SQLException {
	return true;
    }

    /**
     * Provides the catalog name.  For mSQL, this is the database to
     * which you are connected.
     * @return always null
     * @exception java.sql.SQLException never thrown
     */
    public String getCatalog() throws SQLException {
	return database;
    }

    /**
     * @return the encoding for the application
     */
    public String getEncoding() {
	return encoding;
    }
    
    /**
     * @return the input stream for the database connection
     */
    MsqlInputStream getInputStream() {
	return input;
    }
    
    /**
     * Provides meta-data for the database connection.
     * @return the mSQL implementation of DatabaseMetaData
     * @exception java.sql.SQLException this is never thrown
     */
    public DatabaseMetaData getMetaData() throws SQLException {
	return new MsqlDatabaseMetaData(this, log.getLevel());
    }

    /**
     * @return the output stream associated with this connection
     */
    MsqlOutputStream getOutputStream() {
	return output;
    }
    
    /**
     * @return the transaction isolation, always Connection.TRANSACTION_NONE
     * @exception java.sql.SQLException this is never thrown
     */
    public int getTransactionIsolation() throws SQLException {
	return Connection.TRANSACTION_NONE;
    }

    /**
     * @return the type map associated with this connection
     * @exception java.sql.SQLException this is never thrown
     */
    public Map getTypeMap() throws SQLException {
	return typeMap;
    }
    
    /**
     * @return the user name used for this Connection
     * @exception java.sql.SQLException this is never thrown
     */
    public synchronized String getUser() throws SQLException {
	return user;
    }

    /**
     * mSQL does not have warnings.
     * @return always null
     * @exception java.sql.SQLException this is never thrown
     */
    public SQLWarning getWarnings() throws SQLException {
	return null;
    }

    /**
     * @return true if this Connection has been closed
     */
    public synchronized boolean isClosed() {
	return (connection == null);
    }

    /**
     * The connection's I/O streams are currently being used by
     * the connection, one of its statements, or some meta data.
     * @return true if the connection is in use
     */
    synchronized boolean isInUse() {
	return inUse;
    }
    
    /**
     * mSQL does not support read only connections, only read-only access
     * rights.
     * @return false--always
     */
    public boolean isReadOnly() throws SQLException {
	return false;
    }

    /**
     * This gives the driver an opportunity to turn JDBC compliant SQL
     * into mSQL specific SQL.  My feeling is why bother.
     * @exception java.sql.SQLException never thrown
     */
    public String nativeSQL(String sql) throws SQLException {
	return sql;
    }

    /**
     * Callable statements are not supported by mSQL.  This will therefore
     * always throw an exception.
     * @param unused the name of the stored procedure
     * @return nothing, it always throws an exception
     * @exception java.sql.SQLException this will always be thrown
     */
    public CallableStatement prepareCall(String unused) throws SQLException {
	return prepareCall(unused, ResultSet.TYPE_FORWARD_ONLY,
			   ResultSet.CONCUR_READ_ONLY);
    }

    /**
     * Callable statements are not supported by mSQL.  This will therefore
     * always throw an exception.
     * @param unused the name of the stored procedure
     * @param type the type of the procs result sets
     * @param concur the concurrency of the result sets
     * @return nothing, it always throws an exception
     * @exception java.sql.SQLException this will always be thrown
     */
    public CallableStatement prepareCall(String unused, int type, int concur)
    throws SQLException {
	log.log("prepareCall()", MsqlLog.JDBC,
		"Preparing call \"" + unused + "\" " +
		"(type=" + type +",concur=" + concur + ").");
	log.log("prepareCall()", MsqlLog.ERROR,
		"Prepared statements not supported.");
	throw new SQLException("mSQL does not support stored procedures.");
    }
    
    /**
     * Constructs a PreparedStatement matching the specified statement.
     * Note that mSQL does not natively support prepared statements, so
     * mSQL-JDBC prepared statement support is a bit faked.  In other words,
     * there is absolutely no performance benedit to using prepared
     * statements with mSQL.
     * @param sql the prepared statement
     * @return a new PreparedStatement matching the specified SQL
     * @exception java.sql.SQLException the Connection was unable to create
     * a PreparedStatement for the specified SQL
     */
    public PreparedStatement prepareStatement(String sql) throws SQLException {
	return prepareStatement(sql, ResultSet.TYPE_FORWARD_ONLY,
				ResultSet.CONCUR_READ_ONLY);
    }

    /**
     * Creates a prepared statement with the specified type and
     * concurrency for its result sets.
     * @param sql the SQL of the prepared statement
     * @param type the type of the result sets for the prepared statements
     * @param concur the concurrency of the result sets
     * @return a new prepared statement
     * @exception java.sql.SQLException the prepared statement could not be
     * created
     */
    public synchronized PreparedStatement prepareStatement(String sql,
							   int type,
							   int concur)
    throws SQLException {
	PreparedStatement stmt;
	WeakReference ref;
	
	log.log("prepareStatement()", MsqlLog.JDBC,
		"Preparing statement (type=" + type +",concur=" +
		concur + ",sql=\"" + sql + "\").");
	stmt = new MsqlPreparedStatement(this, sql, type, concur,
					 log.getLevel());
	ref = new WeakReference(stmt);
	statements.add(ref);
	return stmt; 
    }

    /**
     * This method notifies the connection that its I/O streams are
     * now released to use by other objects belonging to this connection.
     */
    synchronized void release() {
	log.log("release()", MsqlLog.DRIVER,
		"Connection released by " +
		Thread.currentThread().getName() + ".");
	inUse = false;
	notifyAll();
    }
    
    /**
     * This method always errors since you cannot rollback an mSQL
     * transaction.
     * @exception java.sql.SQLException this will always get thrown by this
     * method
     */
    public void rollback() throws SQLException {
	log.log("rollback()", MsqlLog.JDBC, "Rollback received.");
	log.log("rollback()", MsqlLog.ERROR, "Rollback not supported.");
	throw new MsqlException("mSQL does not support rollbacks.");
    }

    /**
     * This method will thrown an exception if you try to turn auto-commit
     * off since JDBC does not support transactional logic.
     * @param b should always be true
     * @exception java.sql.SQLException thrown if the param is false
     */
    public void setAutoCommit(boolean b) throws SQLException {
	log.log("setAutoCommit()", MsqlLog.JDBC,
		"Setting auto-commit to " + b + ".");
	if( !b ) {
	    log.log("setAutoCommit()", MsqlLog.ERROR,
		    "Auto-commit = false not supported.");
	    throw new MsqlException("mSQL must always be auto-commit = true.");
	}
    }

    /**
     * Selects the database to be used by this connection.
     * @param db the name of the mSQL database
     * @exception java.sql.SQLException failed to set the new database
     */
    public synchronized void setCatalog(String db) throws SQLException {
	String tmp;

	capture();
	log.log("setCatalog()", MsqlLog.MSQL,
		"Sending select database for " + db + ".");
	try {
	    output.writeString("2 " + db, encoding);
	}
	catch( IOException e ) {
	    log.log("setCatalog()", MsqlLog.FATAL,
		    "Failed to send database select: " + e.getMessage());
	    clean();
	    throw new MsqlException(e);
	}
	try {
	    tmp = input.readString(encoding);
	}
	catch( IOException e ) {
	    log.log("setCatalog()", MsqlLog.FATAL,
		    "Failed to receive database select " +
		    "response: " + e.getMessage());
	    clean();
	    throw new MsqlException(e);
	}
	if( tmp.startsWith("-1:") ) {
	    log.log("setCatalog()", MsqlLog.FATAL,
		    "Received error from server: " + tmp);
	    clean();
	    throw new MsqlException(tmp);
	}
	database = db;
	release();
    }

    /**
     * mSQL does not support read-only mode.
     * @param b should only ever be false
     * @exception java.sql.SQLException this is thrown if an attempt is made
     * to put the driver in read-only mode
     */
    public void setReadOnly(boolean b) throws SQLException {
	log.log("setReadOnly()", MsqlLog.JDBC,
		"Setting read-only to " + b + ".");
	if( b ) {
	    log.log("setReadOnly()", MsqlLog.ERROR,
		    "Read-only connections not supported.");
	    throw new MsqlException("mSQL does not support read-only mode.");
	}
    }

    /**
     * This is not supported by mSQL, thus this is a NO-OP.  The
     * transaction isolation level is always set to
     * Connection.TRANSACTION_NONE.
     * @param unused the transaction isolation level
     * @exception java.sql.SQLException this is never thrown
     * @see java.sql.Connection#TRANSACTION_NONE
     */
    public void setTransactionIsolation(int unused)
    throws SQLException {
	log.log("setTransactionIsolation()", MsqlLog.JDBC,
		"Setting transaction isolation to " + unused + ".");
    }

    /**
     * Sets the type map for this connection.
     * @param map the new type map
     * @exception java.sql.SQLException this is never thrown
     */
    public synchronized void setTypeMap(Map map) throws SQLException {
	log.log("setTypeMap()", MsqlLog.JDBC, "Setting type map.");
	typeMap = map;
    }
}
