/* (C) 1999-2000 Samuel Audet <guardia@cam.org>

Profitable use of this source code based on its execution or sale
excluding the cost of the media, shipping, manwork or supporting
hardware is not allowed unless granted by the author himself.  Any
modifications or inclusion of this code in other non-profitable programs
must contain this message and the original author's name. Programs based
on any of this source code must therefore contain the original or
modified source code files.  Use of this source code in a commercial
program will require permission from the author.  No more than 50% of
the original size of the source code from Disk Indexer can be used in a
non-commercial program, unless granted by the author.

*/

import java.io.*;
import java.util.*;


public class HDatabase implements HDatabaseIO
{
   protected HBufferManager dataBuffer = null;
   public static final int DATABASE_VERSION = 1;
   public int databaseVersion = DATABASE_VERSION;

   public final static int PAGE_SIZE = 1024;
   public HDataNode rootNode = null;


   static int numline = 0;


public HDatabase(HBufferManager dataBuffer)
{
   this.dataBuffer = dataBuffer;
   rootNode = dataBuffer.rootNode;
   databaseVersion = dataBuffer.databaseVersion;
}


public final HDataNode findNode(HDataNode parent, String name) throws IOException
{
   int i = 0;
   int count = childCount(parent);
   HDataNode aNode = new HDataNode();
   while(i < count)
   {
      aNode = read(parent,i++);
      if(name.equals(aNode.fileName))
         return aNode;
   }
   return null;
}


public short childCount(HDataNode parent) throws IOException
{
   short count = 0;

   if(parent == null)
      parent = rootNode;

   if(parent.childPointer == -1)
      return 0;

   HDataPage childrenPage = dataBuffer.getPage(parent.childPointer);
   count += childrenPage.amount;

   while(childrenPage.nextBrother != -1)
   {
      childrenPage = dataBuffer.getPage(childrenPage.nextBrother);
      count += childrenPage.amount;
   }

   return count;
}

public HDataNode read(HDataNode parent, int index) throws IOException
{
   HDataPage childrenPage = dataBuffer.getPage(parent.childPointer);

   while(index >= childrenPage.startAt+childrenPage.amount)
   {
      if(childrenPage.nextBrother == -1)
         return null;
      childrenPage = dataBuffer.getPage(childrenPage.nextBrother);
   }

   return childrenPage.getNode(index-childrenPage.startAt);
}


protected int removeRecursionLevel = 0;

public boolean remove(HDataNode aNode) throws IOException
{
   synchronized(rootNode)
   {

   removeRecursionLevel++;

   removeBlob(aNode);

   if(aNode.childPointer != -1)
   {
      int count = childCount(aNode);
      for(int i=0; i < count; i++)
         remove(read(aNode, i)); // removes all children

      HDataPage deletedPage = dataBuffer.getPage(aNode.childPointer); // ughrhfm... ?
      dataBuffer.removePage(aNode.childPointer);
      int nextChild = deletedPage.nextBrother;
      while(nextChild != -1)
      {
         deletedPage = dataBuffer.getPage(nextChild); // ughrhfm... ?
         dataBuffer.removePage(nextChild);
         nextChild = deletedPage.nextBrother;
      }
   }

   // Only move the brother nodes around if its the "top" node
   // from the first call.  Its child are overwritten with blanks,
   // so they don't need to be moved just before being blanked.
   // Also make cleanup of empty pages at end of file
   if(removeRecursionLevel == 1)
   {
      removeFromPage(aNode);
      dataBuffer.trimFile();

      dataBuffer.startLazyWriteTimer();
   }

   removeRecursionLevel--;

   }

   return true;
}


protected boolean removeFromPage(HDataNode aNode) throws IOException
{
   HDataPage page = dataBuffer.getPage(aNode.myPointer);
   int index = page.getIndexOf(aNode);

   if(index != -1)
   {
      page.removeNode(index);

      int nextBrother = page.nextBrother;
      while(nextBrother != -1)
      {
         page = dataBuffer.getPage(nextBrother);
         page.setStartAt(--page.startAt);
         nextBrother = page.nextBrother;
      }

      return true;
   }

   return false;
}

// this needs change so that blob can also be changed
public boolean update(HDataNode aNode, HDataNode updatedNode) throws IOException
{
   synchronized(rootNode)
   {

   HDataPage page = dataBuffer.getPage(aNode.myPointer);
   int index = page.getIndexOf(aNode);

   if(index == -1)
      return false;

   if(page.setNode(updatedNode,index))
   {
      dataBuffer.startLazyWriteTimer();
      return true;
   }
   else
   {
      writeToPage(updatedNode);

      // if node changed page, modify all children header
      if(updatedNode.myPointer != aNode.myPointer && updatedNode.childPointer != -1)
      {
         page = dataBuffer.getPage(updatedNode.childPointer);
         page.setParentPointer(updatedNode.myPointer);

         int nextBrother = page.nextBrother;
         while(nextBrother != -1)
         {
            page = dataBuffer.getPage(nextBrother);
            page.setParentPointer(updatedNode.myPointer);
            nextBrother = page.nextBrother;
         }
      }

      removeFromPage(aNode);
      dataBuffer.startLazyWriteTimer();
   }

   }

   return true;
}


public synchronized boolean move(HDataNode parent, HDataNode aNode) throws IOException
{
   synchronized(rootNode)
   {

   HDataNode nodeToMove = (HDataNode) aNode.clone();
   write(parent, nodeToMove);
   removeFromPage(aNode);
   dataBuffer.startLazyWriteTimer();

   }

   return true;
}


public synchronized boolean write(HDataNode parent, HDataNode aNode) throws IOException
{
   synchronized(rootNode)
   {

   if(parent == null)
      parent = rootNode;

   HDataPage page;

   if(parent.childPointer == -1)
   {
      HDataNode oldPage = (HDataNode) parent.clone();
      page = dataBuffer.createChildPage(parent);
      if(parent != rootNode)
         update(oldPage,parent); // child pointer changed
   }
//   else
//      page = dataBuffer.getPage(parent.childPointer);

   aNode.primaryPointer = parent.childPointer;
   writeToPage(aNode);

   writeBlob(aNode); // blobPointer is set here

   dataBuffer.startLazyWriteTimer();

   }

   return true;
}

protected void writeToPage(HDataNode aNode) throws IOException
{
   HDataPage page = dataBuffer.getPage(aNode.primaryPointer);
   int nodeSize = aNode.size();

   while(!page.addNode(aNode))
   {
      int nextBrother = page.nextBrother;
      if(nextBrother == -1)
         page = dataBuffer.createBrotherPage(page);
      else
         page = dataBuffer.getPage(nextBrother);
   }

   int nextBrother = page.nextBrother;
   while(nextBrother != -1)
   {
      page = dataBuffer.getPage(nextBrother);
      page.setStartAt(++page.startAt);
      nextBrother = page.nextBrother;
   }
}




public HDataNode readParent(HDataNode aChild) throws IOException
{
   HDataNode aNode = new HDataNode();

   if(aChild.parentPointer == 0)
      return rootNode;

   HDataPage page = dataBuffer.getPage(aChild.parentPointer);

   int count = page.getNodeSize();
   for(int i = 0; i < count; i++)
   {
      if(page.getNode(i).childPointer == aChild.primaryPointer)
         return page.getNode(i);
   }

   return null;
}


protected boolean removeBlob(HDataNode aNode) throws IOException
{
   if(aNode.blobPointer == -1)
      return false;

   // loading data

   HDataPage page = dataBuffer.getPage(aNode.blobPointer);

   int nextBrother = page.nextBrother;
   dataBuffer.removePage(page.myPointer);
   while(nextBrother != -1)
   {
      page = dataBuffer.getPage(nextBrother);
      nextBrother = page.nextBrother;
      dataBuffer.removePage(page.myPointer);
   }

   dataBuffer.startLazyWriteTimer();

   return true;
}

protected boolean writeBlob(HDataNode aNode) throws IOException
{
   if(aNode.blobPointer != -1)
      return false;
   byte[] data = aNode.getBlobData();

   if(data == null)
      return false;

   HDataPage page = dataBuffer.createBlobPage(aNode);
   short written = 0;
   short towrite = (short) Math.min(data.length-written, PAGE_SIZE-page.headerSize);
   page.setBlobData(data,written,towrite);
   written += towrite;

   towrite = (short) Math.min(data.length-written, PAGE_SIZE-page.headerSize);
   while(towrite > 0)
   {
      page = dataBuffer.createBrotherPage(page);

      page.setBlobData(data,written,towrite);
      written += towrite;

      towrite = (short) Math.min(data.length-written, PAGE_SIZE-page.headerSize);
   }

   dataBuffer.startLazyWriteTimer();

   return true;
}


public synchronized boolean retrieveBlob(HDataNode aNode) throws IOException
{
   if(aNode.thumbnail != null)
      return true;
   else return readBlob(aNode);
}

protected boolean readBlob(HDataNode aNode) throws IOException
{
   if(aNode.blobPointer == -1)
      return false;

   // finding totalsize

   int totalSize = 0;
   HDataPage page = dataBuffer.getPage(aNode.blobPointer);

   int nextBrother = page.nextBrother;
   totalSize += page.blobData.length;
   while(nextBrother != -1)
   {
      page = dataBuffer.getPage(nextBrother);
      nextBrother = page.nextBrother;
      totalSize += page.blobData.length;
   }

   // loading data

   byte[] data = new byte[totalSize];
   totalSize = 0;
   page = dataBuffer.getPage(aNode.blobPointer);

   nextBrother = page.nextBrother;
   System.arraycopy(page.blobData,0,data,totalSize,page.blobData.length);
   totalSize += page.blobData.length;
   while(nextBrother != -1)
   {
      page = dataBuffer.getPage(nextBrother);
      nextBrother = page.nextBrother;
      System.arraycopy(page.blobData,0,data,totalSize,page.blobData.length);
      totalSize += page.blobData.length;
   }

   return aNode.setBlobData(data);
}



public final static boolean writeBlanks(RandomAccessFile dataFile, int amount) throws IOException
{
   int chunk = amount > 2048 ? 2048 : amount;

   final byte[] blanks = new byte[chunk];
   int i;

   for(i = 0; i < amount/blanks.length; i++)
      dataFile.write(blanks);

   for(i = 0; i < amount%blanks.length; i++)
      dataFile.writeByte(0);

   return true;
}

protected void finalize() throws Throwable
{
   dataBuffer = null;
   rootNode = null;

   super.finalize();
}

}
