/**
 *  Classroom Scheduler
 *  Copyright (C) 2004 Colin Archibald, Ph.D.
 *  https://sourceforge.net/projects/cr-scheduler/
 *
 *  Licensed under the Academic Free License version 2.0
 */

/** Store and maintain a single timeslot.  A timeslot includes the
 *  days of the week
 *  and the start time and stop time for the timeslot.  A time slot can only have
 *  a single start time and stop time for each day.
 *
 * daysOfWeek is a boolean array that corresponds to the days of week in the
 * DateFormatSymbols class.  Sunday is element Calendar.SUNDAY i.e., int 1, etc.
 */

package resources;

import java.io.Serializable;
import java.text.*;
import java.util.*;

public class TimeSlot implements Comparable, Serializable {
    
    static final long serialVersionUID = 911;
    
    private boolean[] daysOfWeek = {false,false,false,false,false,false,false};
    private CRTime start, stop;
    private boolean online = false;
    private boolean tba = false;
    
    public TimeSlot(boolean[] daysOfWeek, CRTime start, CRTime stop) {
        this.daysOfWeek = daysOfWeek;
        this.start = start;
        this.stop = stop;
    }
    /** 
     *  This constructor is only ever used to make a timeslot representing online
     */
    public TimeSlot() {
        online = false;
        start = null;
        stop = null;
    }
    
    public boolean getOnline() {
        return online;
    }
    public boolean getTba(){
        return tba;
    }
    /** 
     * sets this timeslot to be online.
     */
    public TimeSlot setOnline(){
        online = true;
        tba = false;
        daysOfWeek = new boolean[7];
        start = null;
        stop = null;
        return this;
    }
    /**
     *  sets this timeslot to be tba.
     */
    
    public TimeSlot setTba(){
        online = false;
        tba = true;
        daysOfWeek = new boolean[7];
        start = null;
        stop = null;
        return this;
    }
    
    public void setStartTime(CRTime start) {
        this.start = start;
    }
    
    public void setStopTime(CRTime stop) {
        this.stop = stop;
    }
    
    public CRTime getStartTime() {
        return start;
    }
    
    public CRTime getStopTime() {
        return stop;
    }
    
    public boolean[] getDaysOfWeek() {
        return daysOfWeek;
    }
    
    public int getMinutes() {
        return stop.getcrMinutes() - start.getcrMinutes();
    }
    
    /**
     *
     * @return boolean true if the timeSlots overlap
     *
     */
    
    public boolean overlap(TimeSlot secondSlot) {
        if(this.online || this.tba || secondSlot.online || secondSlot.tba){
            return false;
        }
        if (daysOverlap(secondSlot)) {
            if (getStartTime().getcrMinutes() <= secondSlot.getStartTime().getcrMinutes()
            && getStopTime().getcrMinutes() >= secondSlot.getStartTime().getcrMinutes()) {
                return true;
            }
            
            if (getStartTime().getcrMinutes() >= secondSlot.getStartTime().getcrMinutes()
            && getStartTime().getcrMinutes() <= secondSlot.getStopTime().getcrMinutes()) {
                return true;
            }
        }
        return false;
    }
    
    private boolean daysOverlap(TimeSlot secondSlot) {
        for (int i = 0; i < daysOfWeek.length; i++)
            if (daysOfWeek[i] & secondSlot.daysOfWeek[i])
                return true;
        return false;
    }
    
    public String toString() {
        String result = "";
        if (tba) {
            result = " TBA";
        } else if (online){
            result = " ONLINE";
        } else {        
            result = toDaysString();
            result += " " + start.toString() + "-" + stop.toString();
        }
        return result;
    }
    
    // MTWRFS
    
    public String toDaysString() {
        String result = "";
        if(tba){
            result = "TBA";
        } else if (online){
            result = "ONLINE";
        }else{        
            DateFormatSymbols dfs =  new DateFormatSymbols();        
            String[] stringDays = dfs.getWeekdays();

            for (int i = 0; i < daysOfWeek.length; i++) {
                if (daysOfWeek[i])
                    if ( i == Calendar.THURSDAY) result += 'R';
                    else
                        result += stringDays[i].charAt(0);
            }
        }
        return result;
    }
    
    public String toTimeString() {
        
        String result = "";
        if(tba){
            result = "TBA";
        } else if (online){
            result = "NA";
        }else{
            result += " " + start.toString() + "-" + stop.toString();
        }
        return result;
    }
    
    
    public int compareTo(Object obj) {
        if (!(obj instanceof TimeSlot)){
            return 0;
        } 
        TimeSlot slot = (TimeSlot) obj;
        if (tba && slot.tba) return 0;
        if (online && slot.online) return 0;        
        if (tba) return -2;        
        if (online) return -1;
        if (slot.tba) return 2;
        if (slot.online) return 1;
        
        if (obj == null)
            return -1;        
        return start.compareTo(((TimeSlot)obj).start);
    }
    
    // determine if 2 timeSlots are equal - used to avoid duplicates
    
    public boolean equals(Object obj) {
        if (!(obj instanceof TimeSlot)){
            return false;
        }        
        TimeSlot slot = (TimeSlot) obj;
        
        if (tba && slot.tba){
            return true;
        }
        if (online && slot.online){
            return true;
        }
        if (tba || slot.tba || online || slot.online){
            return false;
        }
        
        TimeSlot slot2 = (TimeSlot) obj;
        boolean result = false;
        
        if (start.equals(slot2.start) 
            && stop.equals(slot2.stop) 
            && daysEqual(daysOfWeek, slot2.daysOfWeek))
            return true;
        return false;
    }
    
    private boolean daysEqual(boolean[] days1, boolean[] days2) {
        for(int i = 0; i < days1.length && i < days2.length; i++)
            if (days1[i] != days2[i])
                return false;
        return true;
    }
    
    
    /* test this class */
    
    public static void main(String[] a) {
        CRTime start = null, stop = null;
        boolean days[] = {false,false, true, false, true, true, true, false};
        
        try {
            start = new CRTime(299);
            stop = new CRTime(350);
        }
        catch (CRTimeException crtE) {
            System.out.println(crtE);
        }
        
        TimeSlot slot = new TimeSlot(days, start, stop);
        System.out.println(slot);
    }
}
