/*
 * Summary.java
 *
 * Created on February 16, 2004, 6:40 PM
 */

package application;

import java.io.*;
import java.util.*;
import java.text.*;
import java.awt.*;
import java.awt.print.*;

import resources.*;

/**
 *
 * @author  kyle
 */
public class Summary implements Serializable, Printable, Pageable, Observer {
    static final long serialVersionUID = 911;

    private Hashtable profSummarys;
    private boolean dirty = true;
    private Vector summaryLines;
    
    /** Creates a new instance of Summary */
    public Summary() {
        profSummarys = new Hashtable(10, 0.75f);
        Schedule.getSchedule().addObserver(this);
        buildSummary();
    }
    
    synchronized public void update(Observable sched, Object obj)  {
        buildSummary();       
    }
    
    synchronized public void buildSummary() {
        Schedule schedule = Schedule.getSchedule();
        ArrayList professors = schedule.getProfessors();
        profSummarys.clear();
        dirty = true;
        ProfessorSummary total = new ProfessorSummary(new Professor("All Totals:","", Professor.OTHER));
        ProfessorSummary totalf = new ProfessorSummary(new Professor("Full Time Totals:","", Professor.OTHER));
        ProfessorSummary totala = new ProfessorSummary(new Professor("Adjunct Totals:","", Professor.OTHER));
        
        for(Iterator it = professors.iterator(); it.hasNext(); ) {
            ProfessorSummary summary = new ProfessorSummary((Professor) it.next());
            profSummarys.put(summary.getProfessor(), summary);
            if(summary.getProfessor().getStatus() == Professor.FULL_TIME) {
                totalf.incrementSections(summary.getSections());
                totalf.incrementCreditHours(summary.getCreditHours());
            } else if (summary.getProfessor().getStatus() == Professor.ADJUNCT) {
                totala.incrementSections(summary.getSections());
                totala.incrementCreditHours(summary.getCreditHours());
            }
            total.incrementSections(summary.getSections());
            total.incrementCreditHours(summary.getCreditHours());
        }
        profSummarys.put(ProfessorSummary.TOTAL, total);
        profSummarys.put(ProfessorSummary.TOTAL_FULL, totalf);
        profSummarys.put(ProfessorSummary.TOTAL_ADJUNCT, totala);
    }

    public String calcSummary(){
        String result = "";
        String[] lines = getSummaryLines();
        for(int i = 0; i < lines.length; i++) {
            result += lines[i] + "\n";
        }
        return result;
    }    
    
    // this is a real approximation since capacity is based on the rooms, and the 
    // rooms might not be scheduled yet.
    // Fixed to return the actual count, not sure about the above comment, as the
    // room should be set on the SchedCourse when its created.
    public int totalCapacity(){
        ArrayList courses = Schedule.getSchedule().getSchedCourses();
        int total = 0;
        for(Iterator it = courses.iterator(); it.hasNext(); ) {
            SchedCourse course = (SchedCourse) it.next();
            total += course.getClassroom().getCapacity();
        }
        
        return total;        
    }
    
    /** counts how many full time faculty are in the schedule
     *  @returns the number of full time faculty in the schedule
     *
     */
    public int fullTimeFacultyCount(){
        Schedule schedule = Schedule.getSchedule();
        ArrayList profs = schedule.getProfessors();
        int count = 0;
        for(Iterator it = profs.iterator(); it.hasNext(); ) {
            if(((Professor) it.next()).getStatus() == Professor.FULL_TIME) {
                count++;
            }
        }
        return count;        
    }
    
    /** 
     *  @returns the number of courses taught by full timers 
     */
    public int coursesFullTime(){
        Schedule schedule = Schedule.getSchedule();
        int count = 0;        
        for(Enumeration enum1 = profSummarys.elements(); enum1.hasMoreElements();) {
            ProfessorSummary profSummary = (ProfessorSummary) enum1.nextElement();
            if(profSummary.getProfessor().getStatus() == Professor.FULL_TIME) {
                count += profSummary.getSections();
            }
        }                        
        return count;
    }
    
    public int coursesAdjunct(){
        Schedule schedule = Schedule.getSchedule();
        int count = 0;        
        for(Enumeration enum1 = profSummarys.elements(); enum1.hasMoreElements();) {
            ProfessorSummary profSummary = (ProfessorSummary) enum1.nextElement();
            if(profSummary.getProfessor().getStatus() == Professor.ADJUNCT) {
                count += profSummary.getSections();
            }
        }                        
        return count;                
    }
    
    /** counts how many adjunct faculty are in the schedule
     *  @returns the number of adjunct faculty in the schedule
     *
     */
    public int adjunctFacultyCount(){
        Schedule schedule = Schedule.getSchedule();
        ArrayList profs = schedule.getProfessors();
        int count = 0;
        for(Iterator it= profs.iterator(); it.hasNext(); ) {
            if(((Professor) it.next()).getStatus() == Professor.ADJUNCT) {
                count++;
            }
        }
        return count;
    }
    
    public int totalCreditHours() {
        Schedule schedule = Schedule.getSchedule();
        ProfessorSummary totals = (ProfessorSummary) profSummarys.get(ProfessorSummary.TOTAL);
        return totals.getCreditHours();
    }

    /** creates a summary of what a prof is scheduled for
     *  @returns a one liner String for a particular prof
     */
    private String fullTimerSummary(Professor prof){
        Schedule schedule = Schedule.getSchedule();                
        if(!profSummarys.containsKey(prof)){
            return "Summary for " + prof + " not found!";
        }
        return profSummarys.get(prof).toString();        
    }
    
    synchronized public int getLinesPerPage(PageFormat pageFormat) {                        
        double pageHeight = pageFormat.getPaper().getImageableHeight(); // Height in 1/72" == 1 point       
        int lines = (int) (pageHeight / 9); // rough approximation        
        
        return lines;
    }
    
    // Does this thing work!?!?!? - Kyle
    synchronized public int getNumberOfPages() {
        int linesPerPage = getLinesPerPage(getPageFormat(0));
        int totalLines = getSummaryLines().length;
        return totalLines / linesPerPage; // + 1;  0 means 1 here.
    }
    
    synchronized public int print(Graphics g, PageFormat pageFormat, int pageIndex) {
        if (pageIndex > getNumberOfPages()) {
            return NO_SUCH_PAGE;    // done printing
        }
        
        draw(g, pageFormat, pageIndex,
            new Rectangle((int)(pageFormat.getPaper().getImageableX()),
            (int)(pageFormat.getPaper().getImageableY()),
            (int)(pageFormat.getPaper().getImageableWidth()),
            (int)(pageFormat.getPaper().getImageableHeight())));
        
        return PAGE_EXISTS;        
    }
    
    synchronized private String[] getSummaryLines() {
        
        if(dirty) { // If schedule has changed since last request, rebuild it
            Schedule schedule = Schedule.getSchedule();
            int len = 10 + (profSummarys.size() / 2);
            summaryLines = new Vector(20, 10);                    
            summaryLines.add("Summary of Schedule: " + schedule.getScheduleName());
            summaryLines.add(schedule.updated().trim());
            summaryLines.add("");
            summaryLines.add("Sections:     " + schedule.getSchedCourses().size());
            summaryLines.add("Credit Hours: " + totalCreditHours());
            summaryLines.add("Capacity:     " + totalCapacity());
            summaryLines.add("");         
            summaryLines.add("");
            summaryLines.add("Full Time Faculty Scheduled: " + fullTimeFacultyCount()); 
            summaryLines.add("");
            summaryLines.add("Name                Sections  Credit Hours");
            summaryLines.add("---------------     --------  ------------");
            ProfessorSummary totalsft = (ProfessorSummary) profSummarys.get(ProfessorSummary.TOTAL_FULL);
            ProfessorSummary totalsat = (ProfessorSummary) profSummarys.get(ProfessorSummary.TOTAL);
            
            for(Enumeration enum1 = profSummarys.elements(); enum1.hasMoreElements();) {
                ProfessorSummary profSummary = (ProfessorSummary) enum1.nextElement();
                if(profSummary.getProfessor().getStatus() == Professor.FULL_TIME) {
                    summaryLines.add(profSummary.toString());                    
                }            
            }
            summaryLines.add("---------------     --------  ------------");            
            summaryLines.add(totalsft.toString());      
            summaryLines.add("");
            if(totalsat.getSections() > 0) {
                summaryLines.add("Full Time percentage of all sections: " 
                    + NumberFormat.getPercentInstance().format((double) totalsft.getSections() / totalsat.getSections()));
                summaryLines.add("Full Time percentage of all Credit Hours: " 
                    + " " + NumberFormat.getPercentInstance().format((double) totalsft.getCreditHours() / totalsat.getCreditHours()));
            }
            dirty = false; // Now in synch with profSummary
        }
        summaryLines.trimToSize(); // eliminate any null elements
        return (String[]) summaryLines.toArray(new String[1]); // Forces it to reallocate a new string array
    }

    synchronized private void draw(Graphics g, PageFormat pageFormat, int pageIndex, Rectangle r) {
        Schedule schedule = Schedule.getSchedule();
        // Start paste
        Graphics g2d = g;
        ((Graphics2D)g).setStroke( new BasicStroke(0.5F)); // make thin lines
        
        Font	    titleFont = new Font("Monospaced", Font.BOLD, 10);
        Font	    regFont = new Font("Monospaced", Font.PLAIN, 8);        
        FontMetrics fm = g2d.getFontMetrics(regFont);     // Get font metrics
        int	    lineHeight = fm.getHeight(); // Get line height
        int	    linesPerPage = getLinesPerPage(pageFormat);
        String[]    summary = getSummaryLines();
  
        
        g.setFont(regFont);
        String pageNumberString = (pageIndex + 1) + "/" + getNumberOfPages();
        //        System.out.println("Page#String:" + pageNumberString + ":to here");
        //        System.out.println("is this wide: " + fm.stringWidth(pageNumberString));
        //        System.out.println("r is: " + r);
        // this should print at the bottom of the page... I think
        g.drawString(pageNumberString,
        (int) (r.width - fm.stringWidth(pageNumberString) +
        pageFormat.getImageableX()),
        (int) (pageFormat.getImageableY() + pageFormat.getPaper().getHeight()));               
                        
        int ypos = (int) pageFormat.getImageableY() + lineHeight;
        for(int currentLine = linesPerPage * pageIndex; currentLine < summary.length && currentLine < currentLine + linesPerPage;currentLine++) {
            if(currentLine == 0){
                g.setFont(titleFont);
            } else {
                g.setFont(regFont);
            }
                
            g.drawString(summary[currentLine], (int) pageFormat.getImageableX(), ypos);
            ypos += lineHeight;
        }                                                     
    }    
    
    // Return the Printable object that will render the page
    synchronized public Printable getPrintable(int pageIndex) {
        return this;
    }
    
    // required to implement Pageable  sets the paper to half inch margins and
    //  and imageable area of 7.5 x 10 for max paper useage
    
    synchronized public PageFormat getPageFormat(int pageIndex) {
        // Get the default page format and its Paper object.
        PageFormat pageFormat = PrinterJob.getPrinterJob().defaultPage();
        Paper paper = pageFormat.getPaper();
        pageFormat.setPaper(paper);   // Restore the paper
        return pageFormat;            // Return the page format
    }
}
