

/**
 *  Classroom Scheduler
 *  Copyright (C) 2004 Colin Archibald, Ph.D.
 *  https://sourceforge.net/projects/cr-scheduler/
 *
 *  Licensed under the Academic Free License version 2.0
 */

package application;

import java.awt.event.*;
import java.io.*;
import javax.swing.*;
import javax.swing.filechooser.*;

import resources.*;
/**
 * Handles all the events from the menus fro the Classroom Scheduler Application
 */

public class HandlerMenu implements ActionListener {
    private Schedule schedule = Schedule.getSchedule();
    private JFrame parentWindow;
    
    public HandlerMenu(JFrame parentWindow) {
        this.parentWindow = parentWindow;
    }
    
    public void actionPerformed(ActionEvent e) {
        if (e.getActionCommand() == "Save") {
            if (schedule.getScheduleName().equals(Schedule.DEFAULT_FILE_NAME))
                runSaveDialog("File Save Dialog");    // Display the save dialog
            else
                saveDocument(schedule.getDirectory(),
                schedule.getScheduleName());
        } else if (e.getActionCommand() == "Save As...") {
            runSaveDialog("File Save Dialog");
        } else if (e.getActionCommand() == "Export CSV (spreadsheet)...") {
            runExportCsv();
        } else if (e.getActionCommand() == "New") {
            checkForSave();
            schedule.resetSchedule(null, Schedule.DEFAULT_FILE_NAME);
        } else if (e.getActionCommand() == "Open...") {
            if(schedule != null)
                checkForSave();
            runOpenDialog("Open Schedule");
        } else if (e.getActionCommand() == "Exit") {
            if(checkForSave()) // it is ok to exit
                System.exit(0);
        } else if (e.getActionCommand() == "How to Use") {
            new Overview(parentWindow, true).setVisible(true);
        } else if (e.getActionCommand() == "Example Schedule") {
            makeDummySchedule();
        } else if (e.getActionCommand() == "About")  {
            JOptionPane.showMessageDialog(parentWindow, "Classroom Scheduler " +
            "Version 2.0  Released December 4, 2004\n" +
            "\n Comments to Colin Archibald, PhD\n" +
            "archibald@cfl.rr.com\n\n" +            
            "Released under the Free Academic Lisence 2.0\n",
            "About Classroom Scheduler",
            JOptionPane.INFORMATION_MESSAGE);
        }
        parentWindow.setTitle("Classroom Scheduler: "
        + schedule.getScheduleName() + schedule.updated());
    }
    
    private void runExportCsv()  {
        File csvFile = null;
        
        try  {
            String name = schedule.getScheduleName();
            String csvName;
            int dot;
            
            if ((dot = name.indexOf('.')) != -1)
                csvName = name.substring(0, dot) + ".csv";
            else
                csvName = name + ".csv";
            
            JFileChooser saveDialog = new JFileChooser("Export CSV File");
            CsvFileFilter select = new CsvFileFilter();
        
            saveDialog.addChoosableFileFilter(select);
            saveDialog.setFileFilter(select);            
            saveDialog.setSelectedFile(new File(schedule.getDirectory(), csvName));    // Set the current document name                  
                
            int result = saveDialog.showSaveDialog(parentWindow);    // Display the dialog
        
            // Return if cancel is selected
            if (result == JFileChooser.CANCEL_OPTION)
                return;
        
            // Might want add a warning about overwritting files here.
            
            csvName = saveDialog.getSelectedFile().getName();
            if(csvName != null) {
                csvName = csvName.trim();
                            
                if (!csvName.endsWith(".csv"))
                csvName += ".csv";
                
                csvFile = new File(saveDialog.getCurrentDirectory(), csvName);    // Get the file
                schedule.exportCsv(csvFile);
            }
        } catch(IOException ioe) {
            JOptionPane.showMessageDialog(parentWindow,
            "Failed to write the file" + csvFile + "\n"
            + ioe);
        }
    }
    
    private void runOpenDialog(String dlgTitle) {
        String fileName;
        
        JFileChooser openDialog = new JFileChooser(dlgTitle);
        CrsFileFilter select = new CrsFileFilter();
        
        openDialog.addChoosableFileFilter(select);
        openDialog.setFileFilter(select);
        
        int returnVal = openDialog.showOpenDialog(parentWindow);
        
        if(returnVal == JFileChooser.APPROVE_OPTION) {
            File inputFile = openDialog.getSelectedFile();
            
            if (inputFile != null) {
                loadDocument(inputFile);
            }
        }
    }
    
    /**
     * @param dlgTitle
     * @returns true if the user did not change their mind
     */
    
    private boolean runSaveDialog(String dlgTitle) {
        String	fileName = schedule.getScheduleName();    // get current document name
        JFileChooser saveDialog = new JFileChooser(dlgTitle);
        CrsFileFilter select = new CrsFileFilter();
        
        saveDialog.addChoosableFileFilter(select);
        saveDialog.setFileFilter(select);                                      
        saveDialog.setSelectedFile(new File(schedule.getDirectory(), fileName));    // Set the current document name
        
        int result = saveDialog.showSaveDialog(parentWindow);    // Display the dialog
        
        // Return if cancel is selected
        if (result == JFileChooser.CANCEL_OPTION)
            return false;
        
        fileName = saveDialog.getSelectedFile().getName();    // Get the file name
        
        if (fileName != null) {
            fileName = fileName.trim();
            
            if (!fileName.endsWith(".crs"))
                fileName += ".crs";
            
            // check for overwriting files
            
            File writeThisOne = new
            File(saveDialog.getCurrentDirectory(), fileName);
            if (writeThisOne.exists()) {
                int response = JOptionPane.showConfirmDialog(parentWindow,
                "The file exists. Overwrite?",
                "File Exists",
                JOptionPane.YES_NO_OPTION);
                
                if (response == JOptionPane.NO_OPTION)
                    return false;
            }
            
            schedule.setScheduleName(fileName);		  // Update document name
            
            String directory = saveDialog.getCurrentDirectory().toString();
            
            schedule.setDirectory(directory);    // Save the directory in the document
            parentWindow.setTitle("CPAScheduler" + ": "
            + fileName + schedule.updated());    // Update window title
            saveDocument(directory, fileName);    // Save the document
        }
        return true;
    }
    
    
    /**
     * @param directory
     * @param fileName
     */
    protected void saveDocument(String directory, String fileName) {
        try {
            ObjectOutputStream objectOut =
            new ObjectOutputStream(
            new FileOutputStream(
            new File(directory, fileName)));
            
            objectOut.writeObject(schedule);    // Write the document
            schedule.setChanged(false);
        }
        catch (IOException eOutput) {
            JOptionPane.showMessageDialog(parentWindow,
            "File Write Error" + eOutput.toString());
        }
    }
    
    /**
     * @param fileName
     * @returns true if the user did not cancel
     */
    protected boolean checkForSave() {
        if (schedule.getChanged()) {
            // Prompt to save current document
            int response = JOptionPane.showConfirmDialog(parentWindow,
            "The schedule has changed. Save the changes?",
            "Schedule changed", JOptionPane.YES_NO_CANCEL_OPTION);
            
            if (response == JOptionPane.CANCEL_OPTION)
                return false;
            if (response == JOptionPane.YES_OPTION) {
                
                if (schedule.getScheduleName().equals(
                schedule.DEFAULT_FILE_NAME)) {
                    // Then use Save dialog, but allow the user to change their
                    //  mind
                    if (!runSaveDialog("File Save Dialog")){
                        return false;
                    }
                }else {
                    saveDocument(schedule.getDirectory(),
                    schedule.getScheduleName());
                }
            }
        }
        return true; // it is ok to exit
    }
    
    /**
     * Loads a document from file
     */
    protected void loadDocument(File inputFile) {
        if (inputFile != null) {
            Schedule localSchedule = null;
            try {
                ObjectInputStream objectIn =
                new ObjectInputStream(
                new FileInputStream(inputFile));
                
                localSchedule = (Schedule) objectIn.readObject();
            }
            catch (IOException eInput) {
                JOptionPane.showMessageDialog(parentWindow,
                "File Read Error" + eInput);
            }
            catch (ClassNotFoundException eNoClass) {
                JOptionPane.showMessageDialog(parentWindow,
                "File Read Error - Class not Found?" + eNoClass);
            }
            // Just in case the file got moved after it was saved.
            localSchedule.setDirectory(inputFile.getParent());
            // makes the current schedule the one read
            schedule.resetSchedule(localSchedule, inputFile.getName());
        }
    }
    
    /**
     * Inner class to filter filemames
     * @author
     */
    public class CrsFileFilter extends javax.swing.filechooser.FileFilter {
        private String description;
        
        /**
         * @param pathname
         *
         * @return true if the path is to a folder or a .crs file
         *
         */
        public boolean accept(File pathname) {
            description = new String("Scheduler Files (*.crs)");
            
            if (pathname.isDirectory() ||
            pathname.getName().toLowerCase().endsWith(".crs")) {
                return true;
            }
            return false;
        }
        
        /**
         * @return a description of this filter
         */
        public String getDescription() {
            return description;
        }
    }    // end inner class

    
    // Shamelessly ripped this from CrsFileFilter (but fixed a 
    // typo in the description) - Kyle
    /**
     * Inner class to filter filenames
     * @author
     */
    public class CsvFileFilter extends javax.swing.filechooser.FileFilter {
        private String description;
        
        /**
         * @param pathname
         *
         * @return true if the path is to a folder or a .csv file
         *
         */
        public boolean accept(File pathname) {
            description = new String("Comma Seperated Value Files (*.csv)");
            
            if (pathname.isDirectory() ||
            pathname.getName().toLowerCase().endsWith(".csv")) {
                return true;
            }
            return false;
        }
        
        /**
         * @return a description of this filter
         */
        public String getDescription() {
            return description;
        }
    }    // end inner class
    
    /**
     * Hard code a schedule for testing and development
     */
    
    private void makeDummySchedule() {
        DummySchedule.makeDummySchedule();       
    }
}

