package dks.src.textureEditor;

import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.io.ObjectStreamException;

import javax.imageio.ImageIO;

import com.l2fprod.common.propertysheet.DefaultProperty;
import com.l2fprod.common.propertysheet.Property;

import dks.src.utils.editor.CAbstractEditor;

/**
 * This class implements a PropertySheetTable editor used to modify a texture <br> date : 4 sept. 07
 * @author   DarK Sidious
 */
public class JdksTextureEditorPane extends CAbstractEditor {

	private static final long serialVersionUID = -128409817438958530L;
	public static final String PROPERTY_WIDTH = "propertyWidth";
	public static final String PROPERTY_HEIGHT = "propertyHeight";
	public static final String PROPERTY_TEXTURE = "propertyTexture";
	public static final String PROPERTY_MASK = "propertyMask";

	protected CTexture _textureEditor;

	protected transient PropertyChange _propertyChangeListener;
	protected transient File _texture;
	protected transient File _mask;

	/**
	 * @param shadow the shadow to use
	 */
	public JdksTextureEditorPane(CTexture shadow) {
		_textureEditor = shadow;
		_propertyChangeListener = new PropertyChange();
		this.init();
	}

	protected final void init() {
		addPropertySheetChangeListener(_propertyChangeListener);
		reloadModel();
	}

	/**
	 * @return the shadow to use
	 */
	public CTexture getTextureEditor() {
		return _textureEditor;
	}

	/**
	 * @param texture the file of the texture picture
	 */
	public void setTextureFile(File texture) {
		_texture = texture;
	}

	/**
	 * @param mask the file of the mask picture
	 */
	public void setMaskFile(File mask) {
		_mask = mask;
	}

	/**
	 * @param textureEditor the shadow to use
	 */
	public void setTextureEditor(CTexture textureEditor) {
		_textureEditor = textureEditor;
	}

	/**
	 * Reload the editor with the properties of the texture
	 */
	public void reloadModel() {
		setProperties(new DefaultProperty[0]);
		final DefaultProperty widthProperty = (DefaultProperty) createProperty("Largeur", "Largeur de la texture", Integer.class, _textureEditor.getWidth(), PROPERTY_WIDTH);
		addProperty(widthProperty);
		final DefaultProperty heightProperty = (DefaultProperty) createProperty("Hauteur", "Hauteur de la texture", Integer.class, _textureEditor.getHeight(), PROPERTY_HEIGHT);
		addProperty(heightProperty);
		final DefaultProperty textureProperty = (DefaultProperty) createProperty("Image de texture", "Image de la texture", File.class, _texture, PROPERTY_TEXTURE);
		addProperty(textureProperty);
		final DefaultProperty maskProperty = (DefaultProperty) createProperty("Image du masque", "Image du masque de la texture", File.class, _mask, PROPERTY_MASK);
		addProperty(maskProperty);
	}

	protected void change(Property property) {
		final String name = property.getName();
		if (PROPERTY_WIDTH.equals(name)) {
			_textureEditor.setWidth((Integer) property.getValue());
		} else if (PROPERTY_HEIGHT.equals(name)) {
			_textureEditor.setHeight((Integer) property.getValue());
		} else if (PROPERTY_TEXTURE.equals(name)) {
			_texture = (File) property.getValue();
			if (_texture == null || !_texture.exists()) {
				_textureEditor.setTexture(null);
			} else {
				try {
					final BufferedImage image = ImageIO.read(_texture);
					_textureEditor.setTexture(image);
					_textureEditor.setTextureFileName(_texture.getPath());
				} catch (final IOException e) {
					e.printStackTrace();
				}
			}
		} else if (PROPERTY_MASK.equals(name)) {
			_mask = (File) property.getValue();
			if (_mask == null || !_mask.exists()) {
				_textureEditor.setMask(null);
			} else {
				try {
					final BufferedImage image = ImageIO.read(_mask);
					_textureEditor.setMask(image);
					_textureEditor.setMaskFileName(_mask.getPath());
				} catch (final IOException e) {}
			}
		}
		for (final PropertyChangeListener listener : _listeners) {
			if (!(listener instanceof PropertyChange)) {
				listener.propertyChange(new PropertyChangeEvent(property, property.getName(), property.getValue(), property.getValue()));
			}
		}
	}

	protected class PropertyChange implements PropertyChangeListener {
		public void propertyChange(PropertyChangeEvent arg0) {
			if (arg0.getSource() instanceof Property) {
				change((Property) arg0.getSource());
			}
		}
	}

	public String toString() {
		return "dks.src.textureEditor.JdksTextureEditorPane[texture=" + _texture + "]";
	}

	protected Object readResolve() throws ObjectStreamException {
		_propertyChangeListener = new PropertyChange();
		return this;
	}
}
