package heuser.simpleLogger;

import java.io.File;
import java.io.FileOutputStream;
import java.util.LinkedList;

/**
 * Log messages to standard out or a file. Logged are either errors or normal messages. Incremental logging is turned on by default.
 * This means, calling {@link SimpleLogger#printLog()} or {@link SimpleLogger#printErrors()} returns only the last entries since the preceeding call.
 * @author Jens Heuser, heuserjens@users.sourceforge.net
 */
public abstract class SimpleLogger {

    private static String log = "";
    private static String errors = "";

    private static String logFull = "";
    private static String errorsFull = "";

    private static String errorPrefix = "";

    private static String crlf = System.getProperty("line.separator");

    private static boolean success = true;
    private static boolean incrementalLog = true;
    private static boolean incrementalErr = true;

    private static boolean append = false;

    private static SimpleLoggerPrintStream out = new SimpleLoggerPrintStream(System.out);

    private static LinkedList loggerListenersError  = new LinkedList();
    private static LinkedList loggerListenersMessage= new LinkedList();


    /**
     * Applications that implement SimpleLoggerListener and register here are notified of logged errors.
     * @param sll
     */
    public static void addLoggerListenerError(SimpleLoggerListener sll){
        loggerListenersError.add(sll);
    }

    public static void removeLoggerListenerError(SimpleLoggerListener sll){
        loggerListenersError.remove(sll);
    }


    /**
     * Applications that implement SimpleLoggerListener and register here are notified of logged messages.
     * @param sll
     */
    public static void addLoggerListenerMessage(SimpleLoggerListener sll){
        loggerListenersMessage.add(sll);
    }

    public static void removeLoggerListenerMessage(SimpleLoggerListener sll){
        loggerListenersMessage.remove(sll);
    }


    /**
     * Set logging to incremental. Default is <code>true</code> for both.
     * @param msg Log messages incremental.
     * @param err Log errors incremental.
     */
    public static void logIncremental(boolean msg, boolean err){
        incrementalLog = msg;
        incrementalErr = err;
    }

    /**
     * Set a prefix top be placed at the beginning of every error message.
     * @param prefix The prefix.
     */
    public static void setErrorPrefix(String prefix){
        errorPrefix = prefix;
    }

    /**
     * Whenever a message or error is logged, it will also be written to standard out.<BR>
     * To turn this off, call it with <code>false</code>.
     * @param cout Print every log to console.
     */
    public static void setConsoleOutput(boolean cout){
        out.enableStream(1, cout);
    }

    /**
     *
     * @return Whether logs are printed to standard out or not
     * @see SimpleLogger#setConsoleOutput
     */
    public boolean getConsoleOutput(){
        return out.isEnabledStream(1);
    }


    /**
     * Log a message and -if {@link SimpleLogger#getConsoleOutput} is <code>true</code>- print it to standard out.
     * @param msg The message to log.
     */
    public static void logMsg(String msg){
        log += msg;
        if (incrementalLog) logFull += msg + crlf;
        if (append) out.print(printLog());
            else out.print(printLog());
        append = false;
        log += crlf;

        if (loggerListenersMessage.size() > 0)
            notifyMessageListeners(msg);
    }


    /**
     * Concatenate the message to the last one logged without a new line at console output.
     * @param msg The message to appended to the log.
     */
    public static void logMsgAppend(String msg){
        if (log.endsWith(crlf)) log = log.substring(0, log.length() - 2);
        append = true;
        logMsg(msg);
    }


    /**
     * Log an error and -if {@link SimpleLogger#getConsoleOutput} is <code>true</code>- print it to standard out.
     * @param err The error to log.
     */
    public static void logError(String err){
        success = false;
        errors += errorPrefix + err;
        if (incrementalErr) errorsFull += errorPrefix + err + crlf;
        if (append) out.print(errorPrefix + err);
            else out.println(crlf + errorPrefix + err);
        append = false;
        errors += crlf;

        if (loggerListenersError.size() > 0)
            notifyErrorListeners(err);
    }


    /**
     * Concatenate the error message to the last one logged without a new line at console output.
     * @param err The error message to appended to the error log.
     */
    public static void logErrorAppend(String err){
        if (errors.endsWith(crlf)) errors = errors.substring(0, errors.length() - 2);
        append = true;
        logError(err);
    }

    /**
     * Check, if no errors were logged.
     * @return <code>true</code> if no errors occurred.
     */
    public static boolean success(){
        return success;
    }


    /**
     * Returns the log or only the part of it since the last call of this method.<BR>
     * Depends on whether incremental logging is turned on or not.
     * @return The full or last part of the log.
     * @see SimpleLogger#logIncremental(boolean, boolean)
     */
    public static String printLog(){
        String l = log;
        if(incrementalLog) log = "";
        return l;
    }


    /**
     * Returns the error log or only the part of it since the last call of this method.<BR>
     * Depends on whether incremental logging is turned on or not.
     * @return The full or last part of the error log.
     * @see SimpleLogger#logIncremental(boolean, boolean)
     */
    public static String printErrors(){
        String s = errors;
        if (incrementalErr) errors = "";
        return s;
    }

    /**
     * Returns the complete error log.
     * @return The complete error log.
     */
    public static String printAllErrors(){
        return errorsFull;
    }

    /**
     * Returns the complete message log.
     * @return The complete message log.
     */
    public static String printAllLog(){
        return logFull;
    }


    /**
     * Resets the log. Logging method remains unchanged.
     * @param logs Reset message log.
     * @param errs Reset error log.
     */
    public void resetLog(boolean logs, boolean errs){
        if (logs) {
            log = "";
            logFull = "";
        }

        if (errs){
            errors = "";
            errorsFull = "";
        }
    }


    /**
     * Write log to console <B>and</B> a file. To just log into a file, call {@link #setConsoleOutput(boolean)} before.
     * @param file The file to write the log into.
     * @param fullLog Start logging from scratch or insert full log first.
     */
    public static void logToFile(File file, boolean fullLog){
        try{
            out.add(new FileOutputStream(file));
            if(fullLog){
                out.enableStream(1, false);
                out.print(printAllLog());
                out.enableStream(1, true);
            }
        }catch(Exception e){
            setConsoleOutput(true);
            logError("Error logging to file '" + file.getName() + "'");
        }
    }


    public static void notifyMessageListeners(String msg){
        for (int i = loggerListenersMessage.size() - 1; i >= 0; i--){
            ((SimpleLoggerListener)loggerListenersMessage.get(i)).errorLogged(msg);
        }
    }


    public static void notifyErrorListeners(String err){
        for (int i = loggerListenersError.size() - 1; i >= 0; i--){
            ((SimpleLoggerListener)loggerListenersError.get(i)).errorLogged(err);
        }
    }


    public void finalize(){
        try{
            out.close();
        }catch(Exception e){}
    }
}
