/**
 * $Id: Flatnews.java,v 1.2 2001/09/10 07:24:56 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.server;

import java.io.File;
import java.io.LineNumberReader;
import java.io.InputStreamReader;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import redlight.hotline.HLServer;
import redlight.hotline.HLServerFlatnews;
import redlight.utils.DebuggerOutput;

/**
 * Implements persistent storage for flat news in a human-readable file.
 */
public class Flatnews extends HLServerFlatnews {

    File flatnewsFile;
    long lastModified;

    /**
     * Creates flat news with the given file for persistent storage.
     * @param file the file.
     */
    public Flatnews(HLServer h, File file) {

        super(h);
        flatnewsFile = file;

    }

    /**
     * Called whenever flat news changes.
     */
    public void hasChanged() {

        write();

    }

    /**
     * Load the component from disk if it has changed.
     */
    public void checkModified() {

        if(flatnewsFile.lastModified() != lastModified) {

            Main.logLine("Flat news file " + flatnewsFile.toString() + " changed, reloading.");
            load();

        }

    }

    /**
     * Loads flat news from persistent storage.
     */
    public void load() {

        load(flatnewsFile);
        lastModified = flatnewsFile.lastModified();

    }

    /**
     * Loads flat news from the given InputStream.
     * @param in the InputStream.
     */
    public void load(InputStream in) {

        LineNumberReader lr = null;
        String n = "";

        try {

            lr = new LineNumberReader(new InputStreamReader(in));
            
            for(String line = lr.readLine(); line != null; line = lr.readLine())
                if(line != null)
                    n += line + "\r";

        } catch(Exception e) {

            n = null;

            String error = "Error reading flat news file " + flatnewsFile.toString();

            if(lr != null)
                error += " at line " + lr.getLineNumber();
            
            error += " (" + e.toString() + ").";

            Main.logLine(error);

        }

        if(n != null) {

            news = n;

        }

    }

    /**
     * Writes flat news to persistent storage.
     */
    public void write() {

        write(flatnewsFile);

    }

    /**
     * Writes flat news to the given OutputStream.
     * @param os the OutputStream to write to.
     */
    public void write(OutputStream os) {

        DebuggerOutput.debug("Writing flatnews file, data = " + news);

        try {
            
            os.write(news.getBytes());
            
        } catch(IOException e) {

            Main.logLine("Could not write flat news file " + flatnewsFile.toString() + " (" + e.toString() + ").");
            
        }
        
    }

}

