/**
 * $Id: ResourceFork.java,v 1.1.1.1 2001/07/22 02:44:48 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.macfiles;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.io.File;

import com.apple.MacOS.*;
import com.apple.memory.*;
import com.apple.mrj.*;

import redlight.utils.DebuggerOutput;
import redlight.utils.MacFileUtils;
import redlight.macfiles.MacFile;

/**
 * This class uses JDirect and MRJ class libraries that are 
 * only available under the MRJ on MacOS Classic. This allows
 * direct access to the resource fork on Macintoshes.<p>
 *
 * This code based on something that I think floated around
 * on the mrj-dev mailing list.
 */
public class ResourceFork implements com.apple.NativeObject {    

    private FSSpec spec;
    short refNum;
    private OutputStream output;
    private InputStream input;

    /**
     * Instantiates a ResourceFork object for manipulating
     * the resource fork.
     * @param inFile a reference to the file whose resource fork needs
     *               to be read.
     */
    public ResourceFork(File inFile, 
                        Integer p) throws IOException {

	spec = new FSSpec(inFile);
        String perms;
        byte priv;

        if(((p.intValue() & MacFile.READ_PERM) == MacFile.READ_PERM) && 
           (p.intValue() & MacFile.WRITE_PERM) != MacFile.WRITE_PERM)
            priv = 1;
        else 
            priv = 3;

	if(openResFork(priv) == -1) {

	    createResFork(new MRJOSType(MacFileUtils.getFileCreator(inFile)),
			  new MRJOSType(MacFileUtils.getFileType(inFile)));

	    if(openResFork(priv) == -1)
	       throw new IOException("Open resource fork for "+inFile.getName()+" failed");
	    
	}

	output = new ResourceOutputStream(this);
	input = new ResourceInputStream(this);

    }
 
    /**
     *   Privilege byte:
     *   fsCurPerm      =  0;    {whatever permission is allowed}
     *   fsRdPerm       =  1;    {read permission}
     *   fsWrPerm       =  2;    {write permission}
     *   fsRdWrPerm     =  3;    {exclusive read/write permission}
     *   fsRdWrShPerm   =  4;    {shared read/write permission}
     */
    private short openResFork(byte priv) throws IOException {

	PShort ref = new PShort(); 

	try {

	    MacOSError.CheckResult(FSpOpenRF(spec.getBytes(), priv, ref));

	} catch(MacOSError e) {

	    throw new IOException(e.getMessage());

	}

	refNum = ref.getValue();

	return refNum;

    }
        
    /**
     *  Create the Resource Fork of a file
     *  Note: if openResFork fails (returns -1), and you need a resource fork 
     *  on the file, call createResFork, and try the open again
     */
    private void createResFork(MRJOSType inCreator, MRJOSType inType) {

	FSpCreateResFile(spec.getBytes(), inCreator.toInt(), 
			 inType.toInt(), (short)-1);

    }

    /**
     * @return an OutputStream object for this resource fork.
     */
    public OutputStream getOutputStream() {

	return output;

    }

    /**
     * @return an InputStream object for this resource fork.
     */
    public InputStream getInputStream() {

	return input;

    }

    /**
     * @return the number of bytes available for reading.
     */
    public int available() {

	return length() - getFilePointer();

    }
        
    /**
     * @return the length of the resource fork.
     */
    public int length() {

	PInt r = new PInt(0);
	short err = GetEOF(refNum, r);
	MacOSError.CheckResult(err);

	return r.getValue();

    }

    /**
     * Sets the file pointer to the specified position in 
     * the resource fork.
     * @param pos the position to seek to.
     */
    public void seek(int pos) {

	short err = SetFPos(refNum, (short) 1, pos);
	MacOSError.CheckResult(err);

    }

    /**
     * @return the current file pointer.
     */
    public int getFilePointer() {

	PInt pos = new PInt(0);
	short err = GetFPos(refNum, pos);
	MacOSError.CheckResult(err);
	return pos.getValue();

    }

    /**
     * Closes the resource fork.
     */
    public void close() {

        short err = FSClose(refNum);

        /* Ignore "File not open" errors. */
        if(err != -38)
            MacOSError.CheckResult(err);
        
    }

    /**
     * The JDirect implementation
     */

    /* Specify the library names that we need to work. */

    private static String[] kNativeLibraryNames = { "InterfaceLib" };

    /** Declare the native methods that we're accessing. */

    private static native short SetFPos(short refnum, 
                                        short posmode, 
                                        int pos);
    private static native short GetFPos(short refnum, 
                                        PInt pos);
    private native static short FSpOpenRF(byte[] fsspec, 
                                          byte priv, 
                                          PShort ref);
    private static native short FSClose(short refNum);
    private static native void FSpCreateResFile(byte[] fsspec, 
                                                int creator,
                                                int type, 
                                                short script);
    private static native short GetEOF(short refNum,
                                       PInt logEOF);

}
