/**
 * $Id: Transferrer.java,v 1.5 2001/09/21 02:48:24 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.macfiles;

import java.io.*;
import java.net.Socket;

import redlight.utils.Meter;
import redlight.utils.MeterSource;
import redlight.utils.DebuggerOutput;
import redlight.utils.InterruptableInputStream;

/**
 * Utility class to monitor the transfer progress of Macintosh
 * files by an external Meter object.
 * @see redlight.utils.Meter
 */
public class Transferrer implements MeterSource {
    public boolean interrupted = false;
    boolean isDataFork = true;
    static final int BUF_SIZE = 1024;
    Meter meter;
    long data_len = 0, rsrc_len = 0;
    Thread updater, transferThread;
    long total = 0;
    long resourceBytesWritten = 0, dataBytesWritten = 0;
    IOException transferException;
    Socket xfsocket;
    int xfref;
    int xftask;
    DataInputStream xfinput;
    DataOutputStream xfoutput;

    /**
     * Creates a Transferrer that is monitored by the
     * specified @see Meter.
     * @param task the task id for this transfer.
     * @param ref the reference for this transfer.
     * @param m the Meter object that monitors this transfer.
     * @param s the Socket associated with this transfer.
     */
    public Transferrer(int task, int ref, Meter m) {

        xftask = task;
        xfref = ref;
        transferThread = Thread.currentThread();
	setMeter(m);

    }

    public void setDataInputStream(DataInputStream in) {

        xfinput = in;

    }

    public void setDataOutputStream(DataOutputStream out) {

        xfoutput = out;

    }

    /** 
     * Transfers the data fork.
     * @param in the data fork source data
     * @param out the destination
     * @param length the number of bytes to transfer.
     */
    public void transferDataFork(DataInput in, DataOutput out, long length)
	throws IOException, InterruptedException {
	data_len = length;
	isDataFork = true;
	setDataBytesWritten(0);
	transfer(in, out, length);
    }

    /** 
     * Transfers the resource fork.
     * @param in the resource fork source data
     * @param out the destination
     * @param length the number of bytes to transfer.
     */
    public void transferResourceFork(DataInput in, DataOutput out, long length) throws IOException, InterruptedException {
	rsrc_len = length;
	isDataFork = false;
	setResourceBytesWritten(0);
	transfer(in, out, length);
    }

    /**
     * Creates a thread to update the meter and creates
     * a thread to transfer the data. It waits until 
     * the transfer thread finishes, then checks for
     * any errors.
     */
    private void transfer(DataInput in, DataOutput out, long length) throws IOException, InterruptedException {

        updater = new Thread(new MeterProgress());
        updater.setPriority(Thread.MIN_PRIORITY);
        updater.setName("MeterProgress[" + xftask + "]");
        updater.start();
        
        doTransfer(in, out, length, isDataFork);
        
        updater.interrupt();
        
        try {
            
            updater.join();
            
        } catch(InterruptedException e) {}
        
        if(interrupted)
            throw new InterruptedException("[" + xftask + "] interrupted by " + meter.getClass().getName());
        
        if(transferException != null) {
            
            DebuggerOutput.debug("Transferrer.transfer[" + xftask + "]: throwing transferException: " + transferException.toString());
            throw transferException;
            
        }
        
    }

    /**
     * Returns the number of bytes written for
     * the data fork.
     * @return number of bytes
     */
    public long getDataBytesWritten() {
	return dataBytesWritten;
    }

    /**
     * Returns the number of bytes written for
     * the resource fork.
     * @return number of bytes
     */
    public long getResourceBytesWritten() {
	return resourceBytesWritten;
    }

    /**
     * Sets the number of bytes written to
     * the data fork.
     * @param number of bytes
     */
    public void setDataBytesWritten(long s) {
	dataBytesWritten = s;
    }

    /**
     * Sets the number of bytes written to
     * the resource fork.
     * @param number of bytes
     */
    public void setResourceBytesWritten(long s) {
	resourceBytesWritten = s;
    }

    /**
     * Returns the {@link redlight.utils.Meter} that
     * is monitoring this transfer.
     * @return {@link redlight.utils.Meter} object
     */
    public Meter getMeter() {
	return meter;
    }

    /**
     * Sets the {@link redlight.utils.Meter} 
     * to monitor this transfer.
     * @return {@link redlight.utils.Meter} object
     */
    public void setMeter(Meter m) {
	meter = m;
    }

    public void join() {

        if(transferThread != null) {

            try {

                DebuggerOutput.debug("Transferrer.join: joining transfer thread ...");
                transferThread.join();

            } catch(InterruptedException e) {}

        }

        DebuggerOutput.debug("Transferrer.join: exiting");

    }

    /**
     * This method should be invoked by the {@link redlight.utils.Meter}
     * object that monitors this transfer when the transfer has to be
     * interrupted. This causes the <TT>transferDataFork</TT> and
     * <TT>transferResourceFork</TT> methods to throw an 
     * InterruptedException, which in turn terminates the calling
     * file transfer thread.
     * @see redlight.utils.Meter
     */
    public void interrupt() {

	interrupted = true;

        /* It's very important here that the input and output streams
           are closed, because by closing the streams, we wake up any
           InterruptableInputStream's listening on the input, and this
           enables the transfer thread to die. */

        DebuggerOutput.debug("Transferrer.interrupt[" + xftask + "]: closing i/o streams");

        try {
            
            if(xfinput != null)
                xfinput.close();

        } catch(IOException e) {

            DebuggerOutput.stackTrace(e);

        }

        try {
            
            if(xfoutput != null)
                xfoutput.close();

        } catch(IOException e) {

            DebuggerOutput.stackTrace(e);

        }
        
        DebuggerOutput.debug("Transferrer.interrupt[" + xftask + "]: i/o streams closed");

        if(updater != null) {

            updater.interrupt();
            
            try {
                
                updater.join();
                
            } catch(InterruptedException e) {}

        }

    }

    /**
     * Calls {@link Meter#progressMeter} once a second to allow
     * progress monitoring.
     */
    private class MeterProgress implements Runnable {
	public synchronized void run() {
	    try {
		while(!interrupted) {
                    wait(1000);
                    meter.progressMeter(xfref, total);
		}
	    } catch(InterruptedException e) {}
	}
    }

    /**
     * Copies data from input to output.
     */
    public void doTransfer(DataInput in, DataOutput out, long length, boolean isDataFork) {
        
        DebuggerOutput.debug("Transferrer.doTransfer[" + xftask + "]: started ...");
        try {
            byte[] buf = new byte[BUF_SIZE];
            while(length > 0 && !interrupted) {
                long numberOfBytes = length > buf.length ? 
                    buf.length : length;
                in.readFully(buf, 0, (int) numberOfBytes);
                out.write(buf, 0, (int) numberOfBytes);
                if(isDataFork)			
                    setDataBytesWritten(getDataBytesWritten() + 
                                        numberOfBytes);
                else			
                    setResourceBytesWritten(getResourceBytesWritten() + 
                                            numberOfBytes);
                length -= numberOfBytes;
                total += numberOfBytes;
                
                try {
                    
                    if(DebuggerOutput.on)
                        Thread.currentThread().sleep(100);
                    else
                        Thread.currentThread().yield();
                    
                } catch(InterruptedException _e) {}
            }
        } catch(IOException e) {
            transferException = e;
        }
        
        DebuggerOutput.debug("Transferrer.doTransfer[" + xftask + "]: exiting");
        
    }

}
