package redlight.hotline.iconlib;

import java.io.*;
import java.lang.*;
import java.io.RandomAccessFile;
import java.util.Vector;
import redlight.utils.DebuggerOutput;

/*
                Read the license.txt file for license information!
                If you don't have it, get it at www.MarkusStengel.de.
*/

/**
 *          use this class to open the hotline.dat file, access ICONs, GIFfs, etc.
 *          @author     Markus Stengel
 *          @see          <a href="http://www.MarkusStengel.de">developer site</a>
 */
public class HLExtractor {



     final   int      POS_TABLE_SIZE                      = 0x18;     // table size
     final   int      POS_TABLE_OFFSET                    = 0x1C;     // table entry pos
     final   int      POS_VALID_OFFSET                    = 0x20;     // diff/table offset: validity check
     final   int      POS_OFFSET_UVT_SIZE                 = 0x2C;     // diff/table offset: number of block entries (EMSG, GIFf...)
     final   int      POS_OFFSET_BLOCK_DIFF_EMSG          = 0x30;     // diff/table offset: EMSG
     final   int      POS_OFFSET_BLOCK_DIFF_GIFf          = 0x40;     // diff/table offset: GIFf
     final   int      POS_OFFSET_BLOCK_DIFF_ICON          = 0x50;     // diff/table offset: ICON
     final   int      POS_OFFSET_BLOCK_DIFF_JPEG          = 0xA0;     // diff/table offset: JPEG
     final   int      POS_OFFSET_BLOCK_DIFF_LVT           = 0xB0;     // diff/table offset: Lower Vector Table
     final   int      POS_OFFSET_BLOCK_UVT_DATA           = 0x08;     // diff/identifier to data info
     final   int     SIZE_HLHDRDATA                              = 0x08;     // size of each block signature
     final   int     SIZE_HLTABLE                              = 0x10;     // size of each block entry
     final   int      SIZE_LVTROW                                   = 0x10;     // row size
     final   int     BLOCK_COUNT                                   = 0x08;     // EMSG, GIFf, ... are a total of 8 blocks
     final   int     SIZE_ICON_HEADER                         = 0x40;     // size of the hotline.dat icon-headers
     final   int      SIZE_WIN_BMP_HEADER                    = 0x36;     // size of a standard WIN-BMP header

     int                                         tableSize;
     long                                        tableOffset;
     int                                         dataSize;
     int                                         dataOffset;
     Vector                                      banners;
     int                                         bannersCount;
     RandomAccessFile                            file;
     /**
      *          gives access to the various tables, such as GIFf, ICON, ...
      */
     public HLIndexTable           indexTable;
     /**
      *          <b>true</b> if there are GIFf-entries, <b>false</b> otherwise
      */
     public boolean                    hasGIF;
     /**
      *          <b>true</b> if there are ICON-entries, <b>false</b> otherwise
      */
     public  boolean                    hasICON;
     /**
      *          you should check <a href="#hasGIF">hasGIF</a> before you use this to get access to the GIF-table in the
      *          indexTable structure. <br>
      *          example:<br>
      *          <pre>
      *               HLExtractor hv;
      *               HLTable          ht;
      *               ...
      *               ht = hv.indexTable.get(hv.idGIFTable);
      *               ...
      *          </pre>
      *          <b>value:</b>     <b>-1</b> if there is no table in the indexTable structure, <b>>=0</b> otherwise<br>
      */
     public  int                              idGIFTable          = -1;          // id of the banner-block in the indexTable, s. buildIndexTable
     /**
      *          you should check <a href="#hasICON">hasICON</a> before you use this to get access to the ICON-table in the
      *          indexTable structure. <br>
      *          example:<br>
      *          <pre>
      *               HLExtractor hv;
      *               HLTable          ht;
      *               ...
      *               ht = hv.indexTable.get(hv.idICONTable);
      *               ...
      *          </pre>
      *          <b>value:</b>     <b>-1</b> if there is no table in the indexTable structure, <b>>=0</b> otherwise<br>
      */
     public  int                              idICONTable          = -1;          // id of the icon-block in the indexTable, s. buildIndexTable


     // Assumes the filepointer is at a table block, reads ONE table and returns a populated object.
     // After the operation the pointer is advanced by SIZE_HLTABLE bytes.
      HLTable populateTableUVT(HLTable table)
     throws IOException {
          byte tempData[]      = new byte[SIZE_HLTABLE];                              // allocate the byte field
          if (table == null) table      = new HLTable();                               // allocate the table object if necessary
          file.read(tempData);                                                                      // read the block into the byte field

          table.blockName                =
                    new String(HLUtils.extractBytes(tempData, 0x00, 4));     // extract the block name
          table.dataSizeUVT           =
                    HLUtils.extractInt(tempData, 0x08);                                   // extract the size of the UVT
          table.dataOffsetUVT      =
                    HLUtils.extractInt(tempData, 0x0C);                                   // extract the offset of the UVT

          return table;                                                                                // return the object - just in case...
     }


     // populates the given table with names and identifiers...
      HLTable populateTableUVTNames(HLTable table)
     throws IOException {
          int idCount, identity, i, j, k;
          short strLength;
          HLData tempHLD;
          byte tempData[];
          if (table == null) table      = new HLTable();                               // allocate the table object if necessary
          long currentOffset = file.getFilePointer();                              // store current file position

          file.seek(table.dataOffsetUVT);                                                  // go to the correct file pos
//     DebuggerOutput.debug("current POS: "+ file.getFilePointer());
          idCount = file.readInt();                                                            // read the number of ids
          for (i=0; i<idCount; i++) {                                                       // check out all ids
               tempHLD = new HLData();
               tempHLD.identity      = file.readInt();                                        // read the identity
               strLength = file.readShort();                                                  // read the string(name)-length
               tempData = new byte[strLength];                                             // allocate a byte array for the string length
               file.read(tempData);                                                                 // read the name;
               tempHLD.name          = new String(tempData);                              // store the name in the structure
               table.add(tempHLD);                                                                 // store the new id entry in the table
               j = (int) ((strLength+6)/4)*4;                                             // now we need to align the words correctly
               if (j < (strLength+6)) j += 4;                                             // now we got the total length of the aligned string
               j -= (strLength+6);                                                                 // calculate the amount of bytes to skip
               k = file.skipBytes(j);                                                            // skip to the next id

               if (k != j) {                                                                           // skipping successful?
                    file.seek(currentOffset);                                                  // return to previous file position
                    return null;                                                                      // no, signal error
               }
          }
          file.seek(currentOffset);                                                            // return to previous file position
          return table;
     }

     // assumes the file pointer is at the table offset (NOT the first block entry like EMSG)
     // and starts building the desired block UVTs&LVTs
      HLIndexTable buildIndexTable()
     throws IOException {
          HLIndexTable      index      ;                                                            // allocate the Index-Table-Object
          int                    blockCount;                                                            // how many blocks (EMSG...) are there?
          HLTable               table;                                                                 // later: allocate the object for single tables
          long                    currentOffset     = file.getFilePointer();          // store the current offset;
          int                     i, j, k;                                                                 // temp vars
          byte[][]               blocks;                                                                 // temp field to store the block info
          long                    tempOffset;
          int                    blockPosICON      = -1;
          int                    blockPosGIF       = -1;
          int                    indexTablePos     =  0;
          index           = new HLIndexTable();
//DebuggerOutput.debug("-->buildIndexTable: seeking pos");
          file.seek(currentOffset + POS_OFFSET_UVT_SIZE);                         // go to the row count
          blockCount     = file.readInt();                                                            // get the number of blocks
//DebuggerOutput.debug("-->buildIndexTable: blocks: "+blockCount+" read from pos: " +(currentOffset + POS_OFFSET_UVT_SIZE));
//DebuggerOutput.debug("-->buildIndexTable: current pos:"+file.getFilePointer());

          tempOffset = file.getFilePointer();                                             // store the beginning of the UVT
          blocks      = new byte[blockCount][SIZE_LVTROW];                         // rows are always the same size
          for (i=0; i<blockCount; i++) {                                                  // let's read all the blocks
               file.read(blocks[i]);                                                            // read the blocks
//HLUtils.printField(blocks[i]);
               if (HLUtils.compareBeginStr(blocks[i], "GIFf")) {
                    hasGIF               = true;
                    blockPosGIF      = i;
               }
               if (HLUtils.compareBeginStr(blocks[i], "ICON")) {
                    hasICON               = true;
                    blockPosICON     = i;
               }
          }
          if (hasGIF) {
               file.seek(tempOffset);                                                            // return to the UVT
               file.skipBytes(SIZE_HLTABLE * blockPosGIF);                         // we are at the EMSG, skip it, go to GIFf block
               table = new HLTable();                                                            // create a new table object
               populateTableUVT(table);                                                       // populate the UVT with the GIFf info
               populateTableUVTNames(table);                                                  // populate the GIFf table with the name and identities
               file.seek(tempOffset + blockCount*SIZE_HLTABLE);               // go to the EMSG LVT block
               if (skipToLVTBlock("GIFf", blockCount)) {                              // did we find the block?
                    populateLVTBlock(table);                                                       // populate the GIFf block
                    index.add(table);                                                                 // add the table to the index
                    idGIFTable = indexTablePos;                                             // store the pos in the fast access vars
                    indexTablePos++;                                                                 // increase the indexCounter
               }
          }
          if (hasICON) {
               file.seek(tempOffset);                                                            // return to the UVT
               file.skipBytes(SIZE_HLTABLE * blockPosICON);                         // we are at the EMSG, skip it, go to GIFf block
               table = new HLTable();                                                            // create a new table object
               populateTableUVT(table);                                                       // populate the UVT with the ICON info
               populateTableUVTNames(table);                                                  // populate the ICON table with the name and identities
               file.seek(tempOffset + blockCount*SIZE_HLTABLE);               // go to the EMSG LVT block
               if (skipToLVTBlock("ICON", blockCount)) {                              // did we find the block?
                    populateLVTBlock(table);                                                       // populate the ICON block
                    index.add(table);                                                                 // add the table to the index
                    idICONTable = indexTablePos;                                             // store the pos in the fast access vars
                    indexTablePos++;                                                                 // increase the indexCounter
               }
          }

          file.seek(tempOffset + blockCount*SIZE_HLTABLE);                    // go to the EMSG LVT block
          return index;
     }

     // assumes to be at the beginning of the LVT Blocks, usually EMSG...
      boolean skipToLVTBlock(String name, int blockCount)
     throws IOException {
          int          i, j;
          long          lastOffset;
          long          currentOffset     = file.getFilePointer();                    // store beginning file pos
          byte[]     block = new byte[SIZE_HLHDRDATA];                              // allocate memory for the header

//DebuggerOutput.debug("-->skipToLVTBlock: currentOffset: " + currentOffset);

          for (i=0; i<blockCount; i++) {                                                  // we may need to check all blocks
                    lastOffset = file.getFilePointer();                                   // store the block begin pos
                    file.read(block);                                                                 // read the header of the block
//DebuggerOutput.debug("-->skipToLVTBlock: reading block no. " + i + " data: " + block);
                    if (HLUtils.compareBeginStr(block, name)) {                    // block found
//DebuggerOutput.debug("-->skipToLVTBlock: block " + name + " found!");
                         file.seek(lastOffset);                                                  // return to beginning of block
                         return true;                                                                 // signal success
                    }
                    j = HLUtils.extractInt(block, 4);                                   // extract the row count
                    file.skipBytes(j*SIZE_HLTABLE);                                        // jump to the next block
          }

          file.seek (currentOffset);                                                            // we didn't find it --> return to the beginning file pos
          return false;                                                                                // signal failure
     }

     /** gets an icon by its identity
       *           @param          identity icon identity number
       *          @return          HLIconData object with all necessary data stored in it
       */
     public  HLIconData getIcon(int identity)
     throws IOException {
          long                    currentOffset     = file.getFilePointer();               // store the current offset;
          HLIconData          hlicd                    = new HLIconData();                         // allocate a new HLIconData-object
          byte[]               tempHeader          = new byte[SIZE_ICON_HEADER];          // get the byte field for the header
          HLTable                table;                                                                      // table that is used
          int                    pads;                                                                           // padding var for reading the (possibly) aligned to LW lines correctly
          int                    rawDataSize;                                                            // dimensions of the image = number of pixels (x*y)
          HLData               tempHLD;                                                                      // temporary data storation
          byte[]                tmpBRead;                                                                 // temporary field used for reading the palette
          byte[]                tmpBytes;                                                                 // temporary field used for comfortable reading
          HLRGB                h;                                                                                // temporary object for creating an RGB-Entry object
          int                     i, j, pos,  k, l;

//DebuggerOutput.debug("-->getIcon: gathering the table");
          table      = indexTable.get(idICONTable);                                        // easier access to the table
//DebuggerOutput.debug("-->getIcon: searching for the identity");
          pos           = table.find(identity);                                                       // try to find the icon
          if (pos < 0) {                                                                                     // icon not found
//DebuggerOutput.debug("Error: Icon not found!");                                        // signal error
               return null;                                                                                // stop here
          }
//DebuggerOutput.debug("-->getIcon: icon " + identity + " found at pos " + pos );
//DebuggerOutput.debug("-->getIcon: accessing the object");
          tempHLD = table.get(pos);                                                                 // get the icon info (name, offset, size)
//DebuggerOutput.debug("-->getIcon: name: "  + tempHLD.name + " identity: " + tempHLD.identity + " filePos: " +tempHLD.dataOffset+ " dataSize: " + tempHLD.dataSize);
//DebuggerOutput.debug("-->getIcon: jumping to offset " + tempHLD.dataOffset);
          file.seek(tempHLD.dataOffset);                                                       // jump to the correct pos
//DebuggerOutput.debug("-->getIcon: reading the header");
          file.read(tempHeader);                                                                      // read the header

//DebuggerOutput.debug("-->getIcon: extracting header info");
          hlicd.width                    = HLUtils.extractInt(tempHeader, 0x10);     // extract the width
          hlicd.height               = HLUtils.extractInt(tempHeader, 0x14);     // extract the height
          hlicd.widthInBytes     = HLUtils.extractInt(tempHeader, 0x18);     // extract the width in bytes
          hlicd.palLengthBytes     = HLUtils.extractInt(tempHeader, 0x24);     // extract the palette length in bytes
          hlicd.transparency     = (HLUtils.extractInt(tempHeader, 0x2C));     // extract the transparency color
          hlicd.palLengthLW          = HLUtils.extractInt(tempHeader, 0x30);     // extract the palette length in long words
//DebuggerOutput.debug("-->getIcon: width: " + hlicd.width + " widthInBytes: " + hlicd.widthInBytes);

          pads = hlicd.widthInBytes - hlicd.width;                                        // padding necessary?
          if (pads < 0) pads = 0;                                                                      // just in case there is an error in the file...
//DebuggerOutput.debug("-->getIcon: gathering palette memory, pads are at " + pads);
          hlicd.palette                = new byte[hlicd.palLengthLW*3][3];               // allocate the memory
          tmpBRead                     = new byte[4];
          for (i=0; i<hlicd.palLengthLW; i++) {                                             // read the entire palette
               file.read(tmpBRead);
               hlicd.palette[i][0]     = tmpBRead[0];                                             // copy red
               hlicd.palette[i][1]     = tmpBRead[1];                                             // copy green
               hlicd.palette[i][2]     = tmpBRead[2];                                             // copy blue
          }
          // palette done, now the data stuff
          rawDataSize                         = hlicd.width  * hlicd.height;               // calculate the data of the raw data block
          hlicd.rawData                    = new byte[rawDataSize];                         // calculate the required size of the data byte field
          tmpBytes                          = new byte[hlicd.width+pads];                    // allocate data structure for comfortable reading
          for (i=0; i< hlicd.height; i++) {                                                  // read linewise so we can take care of the pads
               file.read(tmpBytes);                                                                      // read the raw data
               HLUtils.copyBytes(hlicd.rawData,
                                             tmpBytes,
                                             i*hlicd.width,
                                             0,
                                             hlicd.width);                                                  // copy the stuff into the structure
          }
//          hlicd.data                    = new Vector();                                             // create the vector of objects objects for storing the data
          hlicd.malloc(rawDataSize);                                                              // get the necessary memory to store the data     
          for (i=0; i<rawDataSize; i++) {                                                       // now we store the complete (expanded) data in the structure
               j = HLUtils.signedToUnsigned(hlicd.rawData[i]);                         // get the coded color at pos i
//                    h = new HLRGB((byte) 0, (byte) 0, (byte) 0);
                    if (j>=hlicd.palLengthLW) {                                                  // ISSUE: QUICK HACK: either file spec is wrong or this means a corrupted image
DebuggerOutput.debug(" ICON USES ILLEGAL PALETTE COLOR OR THIS IS AN ISSUE TO BE FIXED! identity: " + identity);
                         return null;
                    }
                    else {
                         tmpBytes = hlicd.palette[j];
                    }
                    hlicd.add(tmpBytes[0], tmpBytes[1], tmpBytes[2]);                                                                           // add it to the vector
          }
          file.seek(currentOffset);                                                                 // return to the previous pos
          return hlicd;                                                                                     // return the icon data
     }

     // assumes to be at the beginning of a LVT block, gathers and matches the data and stores it into the
     // given table
      boolean populateLVTBlock(HLTable table)
     throws IOException {
          HLDataHdr          hdr = new HLDataHdr();                                        // allocate signature block object
          long                    currentOffset     = file.getFilePointer();          // store the current offset;
          byte[]               tempField = new byte[SIZE_HLHDRDATA];               // byte field where the signature is read into
          int                     i, j, k;                                                                 // temp vars
          HLData               data;

          file.read(tempField);                                                                 // read the signature
//DebuggerOutput.debug("-->parseLVT: read signature: " + tempField);
          hdr = HLUtils.extractDataFieldHLDataHdr(tempField);               // extract it and put it into the hdr-object
          if (table.blockName.compareTo(hdr.blockName) != 0) {               // security check: do UVT/current block match?
               file.seek (currentOffset);                                                       // mismatch --> return to previous offset
//DebuggerOutput.debug("-->parseLVT: aborting extraction! blockName: " + hdr.blockName);
               return false;                                                                           // signal error by returning null
          }
          tempField = new byte[SIZE_LVTROW];                                             // now we need the array of the size of a row
          for (i=0; i<hdr.rowCount; i++) {                                                  // now add identity, offset and size for each block item
               file.read(tempField);                                                            // read the data
               data = HLUtils.extractDataFieldHLData(tempField);               // extract identity, offset and size into the data object
               k = table.find(data.identity);                                             // find the identity pos
               if (k<0) {                                                                                // identity not found --> icon is unnamed
                    data.name = new String("unnamed"+data.identity);          // give it an automatic name
                    table.add(data);                                                                 // store the new data in the table
               }
               else {                                                                                     // identity found
                    data.name = table.get(k).name;                                        // store the proper icon name in the object
                    table.set(k, data);                                                            // store the modified data in the table
               }
//DebuggerOutput.debug("-->parseLVT: name: " + data.name + " identity: " + data.identity + " dataOffset: " + data.dataOffset);
          }
          return true;                                                                                // signal success
     }


     /** create a bitmap from a HLIconData object
       *     @param          hlicd     icon data stored in a HLIconData object
       *     @return          windows bitmap as a byte field
       */
     public  byte[] createWinBitmap(HLIconData hlicd) {
          final  int           SIZE_WIN_BMP_HEADER               = 0x36;                    // size of a standard WIN-BMP header
          final  int          SIZE_PIXEL_IN_BYTES               = 0x03;                    // size of a pixel in bytes: RGB --> 3  bytes
          final  int          FILE_TYPE                              = 0x4D42;               // file type, USE: 2 bytes
                       int          size;                                                                      // size, yet to be calculatet, use: 4
          final  int          X_HOTSPOT                              = 0x0000;               // reserved 0, USE: 2
          final  int          Y_HOTSPOT                              = 0x0000;               // reserved 1, USE: 2
          final  int          OFFSET_BITS                              = 0x00000036;          // offset bytes, USE: 4
          final  int          SIZE_HEADER_STRUC                    = 0x00000028;          // size of a part of the header, USE: 4
                    int          widthInPix;                                                            // yet to be calculated, use: 4
                    int          heightInPix;                                                       // yet to be calculated, use: 4
          final  int        PLANES_COUNT                         = 0x0001;               // number of planes, USE: 2
          final  int          BITS_PER_PIXEL                         = 0x0018;               // number of bits/pixel, USE: 2
          final  int          COMPRESSION_TYPE                    = 0x00000000;          // no compression, USE: 4
                    int          bitmapSize;                                                            // image size, USE: 4
          final  int          XPELS_PER_METER                    = 0x00000EC3;          // target device resolution, USE: 4
          final  int          YPELS_PER_METER                    = 0x00000EC3;          // target device resolution, USE: 4
          final  int           BITS_PER_COLOR_USED               = 0x00000000;          // how many bits are used, USE: 4
          final  int           NUMBER_IMPORTANT_COLORS          = 0x00000000;          // all colors are important, USE: 4
                    int          pads;                                                                      // padding var for creating a correctly aligned BMP
                    int          widthInBytes;                                                       // the real width that is required by a color entry
                    byte[][] koord;                                                                 // coord. system where the decoded image is stored in RGB-triples
                    HLRGB           tempRGB;                                                                 // temporary RGB-Object to easen access to the proper RGB entry
                    int          pos;                                                                      // field pointer used for copying the image from the coord. system into the BMP structure
                    byte[]      data;                                                                      // here the COMPLETE bmp image is stored
                    int           i, j, k;                                                                 // temp vars

          if (hlicd == null) {                                                                      // no object data provided
               return null;                                                                           // therefore we can't do anything
          }

          widthInBytes= hlicd.width * SIZE_PIXEL_IN_BYTES;                    // calculate the size of a color entry (24bits/pixel = 3 bytes)
          j                    = (widthInBytes / 4) * 4;                                        // find out if it needs to be aligned
          if (j<widthInBytes)
               pads= (j+4-widthInBytes);                                                        // yes, padding is necessary
          else
               pads = 0;                                                                                // no, padding is unnecessary

          bitmapSize     = hlicd.height * (widthInBytes+pads);                    // calculate the bitmap data size
          size                = bitmapSize + SIZE_WIN_BMP_HEADER;                         //     calculate the complete size
          widthInPix     = hlicd.width;                                                            // icon width
          heightInPix     = hlicd.height;                                                       // icon height
          data                = new byte[size];                                                       // get the necessary space
          // start with BMP header creation, look up the BMP spec for more details
          HLUtils.storeReversed(data, HLUtils.intToBytes(FILE_TYPE),                                 0, 2, 2);
          HLUtils.storeReversed(data, HLUtils.intToBytes(size),                                      2, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(X_HOTSPOT),                                 6, 2, 2);
          HLUtils.storeReversed(data, HLUtils.intToBytes(Y_HOTSPOT),                                8, 2, 2);
          HLUtils.storeReversed(data, HLUtils.intToBytes(OFFSET_BITS),                          10, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(SIZE_HEADER_STRUC),               14, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(widthInPix),                          18, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(heightInPix),                          22, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(PLANES_COUNT),                          26, 2, 2);
          HLUtils.storeReversed(data, HLUtils.intToBytes(BITS_PER_PIXEL),                    28, 2, 2);
          HLUtils.storeReversed(data, HLUtils.intToBytes(COMPRESSION_TYPE),                30, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(bitmapSize),                          34, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(XPELS_PER_METER),                     38, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(YPELS_PER_METER),                     42, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(BITS_PER_COLOR_USED),           46, 0, 4);
          HLUtils.storeReversed(data, HLUtils.intToBytes(NUMBER_IMPORTANT_COLORS),     50, 0, 4);

          // now comes the data
          pos      = SIZE_WIN_BMP_HEADER;                                                       // we start directly after the header block
          k          = 0;                                                                                     // initialize k to start with the first coded color entry
          koord = new byte[hlicd.height][widthInBytes];                         // allocate memory for the image
          // ISSUE: I don't know why, but I had to swap the image - maybe i am overlooking sth.
          for (i = hlicd.height-1; i >= 0; i--) {                                   // do all the rows
               for (j = 0; j < hlicd.width;  j++) {                                   // do all row entries
                    koord[i][j*3 + 0]= hlicd.getBlue(k);                              // store the blue component
                    koord[i][j*3 + 1]= hlicd.getGreen(k);                              // store the green component
                    koord[i][j*3 + 2]= hlicd.getRed(k);                              // store the red component
                    k++;                                                                                     // advance to the next raw data entry
               }
          }
          k           = 0;                                                                                     // reinitialize k, use as a dest. field pointer for copying the image into the BMP data table
          for (i = 0; i < hlicd.height;i++) {                                             // do the rows
               for (j=0; j < hlicd.width; j++) {                                        // do the row entries
                    HLUtils.copyBytes(data,
                                                  koord[i],
                                                  pos + k,
                                                  j * SIZE_PIXEL_IN_BYTES,
                                                  SIZE_PIXEL_IN_BYTES);                          // copy each pixel (size SIZE_PIXEL_IN_BYTES) into the BMP structure
                    k += SIZE_PIXEL_IN_BYTES;                                                  // advance the pointer
               }
               pos+=pads;
          }
          return data;
     }
     /** validate whether the given file is a correct hotline.dat file
       *          @param     tableOffset vector tables offset
       *          @return     <b>true</b> if it is a valid hotline.dat file, <b>false</b> otherwise
       */
     public  boolean isHotline(long tableOffset)
     throws IOException {
          long          currentOffset     = file.getFilePointer();                    // store the current file position
          byte[]     position           = new byte[4];                                        // allocate memory so we can read the position data easily
          long          readOffset;                                                                      // used to verify the positions

          if (tableOffset >= file.length()) {                                             // file to small or invalid table offset
               return false;                                                                           // so it can't be a valid file
          }

          file.seek (tableOffset + POS_VALID_OFFSET);                              // go to the hotline.dat validity position
          file.read (position);                                                                 // read the (hopefully) correct position

          readOffset      = HLUtils.signedToUnsigned(position[0]) * 0x1000000
                              + HLUtils.signedToUnsigned(position[1]) * 0x10000
                              + HLUtils.signedToUnsigned(position[2]) * 0x100
                              + HLUtils.signedToUnsigned(position[3]);               // convert the byte-field to a long-value
          file.seek (currentOffset);                                                            // now reset the file pointer
          if ((tableOffset+POS_VALID_OFFSET) == readOffset) {               // file valid?
               return true;                                                                           // yes
          }
          else {
               return false;                                                                           // no, file is invalid
          }
     }

     /** extract the tables size from the hotline file
       *      @return          vector tables size
       */
     public  int getTableSize()
     throws IOException {
          long          currentOffset;
          int          tableSize;

          currentOffset      = file.getFilePointer();                                   // store current offset
          file.seek(POS_TABLE_SIZE);                                                            // jump to the pos. in the file where it is stored
          tableSize          = file.readInt();                                                  // read the size of the table
          file.seek(currentOffset);                                                            // reset file pointer

          return     tableSize;                                                                      // return the table size
     }
     /** extract the tables offset from the hotline file
       *      @return          vector tables offset
       */
     public  int getTableOffset()
     throws IOException {
          long          currentOffset;
          int          tableOffset;

          currentOffset      = file.getFilePointer();                                   // store current offset
          file.seek(POS_TABLE_OFFSET);                                                       // jump to the pos. in the file where it is stored
          tableOffset          = file.readInt();                                                  // read the offset of the table
          file.seek(currentOffset);                                                            // reset file pointer

          return     tableOffset;                                                                 // return the table offset
     }

     /** <i><u>quick start function</u></i> <br>
       * note: use of this function is highly recommended - it does the whole job for you <br>
       * example:<br>
       *     <pre>
       *           hv = new HLExtractor();
       *           if (!hv.quickStart("hotline.dat")) {
       *                return;
       *           }
       *           // now an icon extraction
       *           hlid = hv.getIcon (128);
       *           if (hlid != null) {
       *                fpos      = hv.indexTable.get(hv.idICONTable).find(ic);
       *                hld      = hv.indexTable.get(hv.idICONTable).get(fpos);
       *                if (hld == null)
       *                {
       *                     DebuggerOutput.debug("No object is at that pos!");
       *                }
       *                else {
       *                     wbmp      = hv.createWinBitmap(hlid, hld);
       *                     tname = new String("Icon"+(int) hld.getIdentity());
       *                     tname=tname.concat(".bmp");
       *                     tempFile = new FileOutputStream(tname);
       *                     tempFile.write(wbmp);
       *                     tempFile.close();
       *                }
       *           }
       *           hv.close();
       *     </pre>
       *      @return          <b>true</b> if successful, <b>false</b> when something went wrong
       */
     public  boolean quickStart(String hotlineFile)
     throws IOException {
          if (!open(hotlineFile)) {                                                            // open the hotline.dat file
               return false;                                                                           // something went wrong
          }
          tableSize      = getTableSize();                                                       // get the table size
          tableOffset = getTableOffset();                                                  // get the table offset
          if (!isHotline(tableOffset)) {                                                  // is it a valid hotline file?
               file.close();                                                                           // no, close file
               return false;                                                                           // report it
          }
          file.seek(tableOffset);                                                                 // everything went fine, go to the table offset
          indexTable = buildIndexTable();                                                  // build the index table

          return true;
     }

     /**     open a hotline.dat file, reference is stored internally
      *          @param     name     full path and name to the hotline.dat file
      *          @return     <b>true</b> if the file was successfully opened, <b>false</b> otherwise
      */
     public  boolean open(String name)
     throws IOException {
          file                = new RandomAccessFile(name, "r");                         // open the hotline.dat file
          if (file == null) {                                                                      // successfully opened?
               return false;                                                                           // no
          }
          else {
               return true;                                                                           // yes
          }
     }
     /** closes the internally opened hotline.dat file */
     public  void close()
     throws IOException {
          if (file != null) {                                                                      // is it opened?
               file.close();                                                                           // yes, close it
          }
     }

     /** extracts a gif (-banner) from the hotline.dat-file
       *     @param     identity     identity of the GIF-images
       *     @return     GIF-image stored in a byte field
       */
     public  byte[] getGIF (int identity)
     throws IOException {
          long           currentOffset;
          byte[]     data;
          HLData     hld;
          int          tablePos;
          HLTable      table;

          table                    = indexTable.get(idGIFTable);                    // easen access to the table
          tablePos               = table.find(identity);                                        // find the pos. of the wanted gif-file
          if (tablePos < 0) {                                                                      // identity not found
               return null;                                                                           // signal with the return value
          }
          hld                    = table.get(tablePos);                                        // easen access
          data                    = new byte[hld.dataSize];                                   // allocate memory for the GIF
          currentOffset     = file.getFilePointer();                                   // store the current file pos.

          file.seek (hld.dataOffset);                                                       // go to the file pos of the GIF
          file.read (data);                                                                           // read it
          file.seek (currentOffset);                                                            // return to previous file pos.

          return data;                                                                                // return the GIF
     }




}
