/**
 * $Id: TransferThread.java,v 1.6 2001/10/09 04:48:51 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.hotline;
/**
 * $Id: TransferThread.java,v 1.6 2001/10/09 04:48:51 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

/* FIXME: Not all of these imports are necessary. */

import java.net.ServerSocket;
import java.net.Socket;
import java.net.InetAddress;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Random;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.Serializable;
import java.io.File;
import java.io.DataOutputStream;
import java.io.DataInputStream;
import java.io.ByteArrayOutputStream;

import redlight.utils.InterruptableInputStream;
import redlight.utils.DebuggerOutput;
import redlight.utils.ToArrayConverters;
import redlight.utils.QuickSort;
import redlight.utils.Meter;
import redlight.utils.MeterSource;
import redlight.utils.MacFileUtils;
import redlight.utils.FilenameUtils;
import redlight.utils.TimeFormat;
import redlight.utils.BytesFormat;
import redlight.macfiles.MacFile;
import redlight.macfiles.Transferrer;

/**
 * One of these is created for each file transfer.
 */
class TransferThread extends Thread implements Meter {

    Socket client;
    HLTransferServer transferServer;
    HLServer.TransferRequest transferRequest = null;
    long creationTime = 0;
    boolean isAlive = false;
    private KillThread killThread;

    TransferThread(HLTransferServer transferServer, Socket client) {

        this.transferServer = transferServer;
        this.client = client;

        killThread = new KillThread(client);
        killThread.start();

        synchronized(transferServer.transferThreads) {

            transferServer.transferThreads.addElement(this);
            
        }

        this.start();

    }

    class KillThread extends Thread {

        Integer lock = new Integer(0);
        boolean running = false;
        Socket client;

        KillThread(Socket client) {

            this.client = client;

        }

        public void disconnect() {

            synchronized(lock) {

                if(running)
                    return;

            }

            synchronizedDisconnect();

        }

        private synchronized void synchronizedDisconnect() {

            DebuggerOutput.debug("TransferThread[" + transferRequest + "]: notifying killthread");
            DebuggerOutput.debug("Notifying KillThread ...");
            notify();

        }
        

        public synchronized void run() {
            
            DebuggerOutput.debug("KillThread started");
            
            try {

                DebuggerOutput.debug("KillThread sleeping ...");
                wait();
                
            } catch(InterruptedException e) {}
            
            synchronized(lock) {
                
                running = true;
                
            }
            
            DebuggerOutput.debug("KillThread running");
            
            DebuggerOutput.debug("TransferThread[" + transferRequest + "]: killthread running...");
            
            DebuggerOutput.debug("TransferThread[" + transferRequest + "]: closing output stream");
            
            try {
                
                client.getOutputStream().close();
                
            } catch(IOException e) {
                
                DebuggerOutput.stackTrace(e);
                
            } catch(NullPointerException e) {

                DebuggerOutput.stackTrace(e);

            }
            
            DebuggerOutput.debug("TransferThread[" + transferRequest + "]: closing input stream");
            
            try {
                
                client.getInputStream().close();
                
            } catch(IOException e) {
                
                DebuggerOutput.stackTrace(e);

            } catch(NullPointerException e) {

                DebuggerOutput.stackTrace(e);
                
            }
            
            DebuggerOutput.debug("TransferThread[" + transferRequest + "]: closing socket");
            
            try {
                
                client.close();
                
            } catch(IOException e) {
                
                DebuggerOutput.stackTrace(e);
                
            }
            
            DebuggerOutput.debug("TransferThread[" + transferRequest + "]: interrupting thread");
            
            try {
                
                nextInLine();
                this.interrupt();
                DebuggerOutput.debug("TransferThread[" + transferRequest + "]: joining thread");
                join();
                
            } catch(InterruptedException e) {
                
                DebuggerOutput.stackTrace(e);
                
            }
            
            DebuggerOutput.debug("TransferThread[" + transferRequest + "]: killthread done");
            
            /* Release transfer reference for reuse. */
            
            if(transferRequest != null)                
                transferServer.hls.getTransferQueue().destroy(transferRequest.ref);

            transferServer.hls.log(client.getInetAddress(),
                                   "File transfer connection closed.");
            
            synchronized(transferServer.transferThreads) {
                
                transferServer.transferThreads.removeElement(this);

            }
            
            DebuggerOutput.debug("KillThread exiting");

        }
        
    }

    public void disconnect() {

        killThread.disconnect();

    }

    public void run() {

        DebuggerOutput.debug("TransferThread starting ...");

        DataInputStream xfinput = null;
        DataOutputStream xfoutput = null;
        HLProtocol.FileTransferHeader fth = null;

        creationTime = System.currentTimeMillis();
        transferServer.hls.log(client.getInetAddress(), 
                               "Accepted file transfer connection");

        try {

            if(isInterrupted())
                throw new InterruptedException();

            /* Wait for the client to start sending us the 
               file transfer header, then look up the requested
               transfer ID. */

            xfinput = new DataInputStream(new InterruptableInputStream(client.getInputStream()));
            xfoutput = new DataOutputStream(client.getOutputStream());
            fth = transferServer.hls.hlp.new FileTransferHeader(xfinput);

            DebuggerOutput.debug("TransferThread[" + fth.ref + "]: " + fth.toString());

            transferRequest = 
                transferServer.hls.getTransferQueue().get(fth.ref);

            DebuggerOutput.debug("TransferThread[" + fth.ref + "]: transferRequest = " + transferRequest);

            if(transferRequest == null) {

                transferServer.hls.log(client.getInetAddress(), "Cannot find transfer request ID " + fth.ref + ".");
                return;

            }
            
            DebuggerOutput.debug("TransferThread[" + fth.ref + "]: transferRequest OK");
            
            /* Reference exists; the client has us convinced so
               far. */
            
            isAlive = true;
            transferRequest.transferThread = this;
            
            if(transferRequest.cancelled == true) {
                
                DebuggerOutput.debug("TransferThread[" + fth.ref + "]: transferRequest CANCELLED");
                throw new InterruptedException("TransferThread[" + fth.ref + "]: transferRequest cancelled");
                
            }
            
            standInLine();
            
            if(transferRequest.type == HLServer.TransferRequest.FILE_UPLOAD) {
                
                DebuggerOutput.debug("TransferThread[" + fth.ref + "]: Receiving file ...");
                
                /* Prepare to receive a file. */
                
                client.setReceiveBufferSize(HLProtocol.SOCK_RECEIVE_BUF_SIZE);
                
                HLProtocol.FileTransferInfo fti = 
                    transferServer.hls.hlp.new FileTransferInfo(xfinput);
                
                DebuggerOutput.debug("TransferThread[" + fth.ref + "]: " + fti.toString());
                
                /* Create a transferrer to handle the actual
                   transfer and inform us about progress. */
                
                Transferrer transferrer = new Transferrer(0, fth.ref, this);
                transferrer.setDataInputStream(xfinput);
                transferrer.setDataOutputStream(xfoutput);
                
                startMeter(transferrer, fth.ref, fti.fileName, fth.len - (fti.data.length + 16 /* HTXF header length */));
                
                /* Initialize the local file. */
                
                DebuggerOutput.debug("TransferThread[" + fth.ref + "]: " + "Initializing local file");
                
                if(transferRequest.local.getDataSize() == 0)
                    transferRequest.local.writeHeader(fti.fileName, 
                                                      fti.fileType,
                                                      fti.fileCreator,
                                                      fti.fileComment,
                                                      fti.creationDate,
                                                      fti.modificationDate,
                                                      fti.finderFlags);
                
                /* Get the data fork size. Should check
                   whether magic matches "DATA" or whatever. */
                
                byte[] l = new byte[8];
                xfinput.skipBytes(8); /* Magic is first 4 bytes. */
                xfinput.readFully(l, 0, 8);
                long size = ToArrayConverters.byteArrayToLong(l);
                
                DebuggerOutput.debug("TransferThread[" + fth.ref + "]: " + "Receiving data fork (" + size + " bytes).");
                
                /* Seek to end of local file (i.e. resume). */
                
                transferRequest.local.getDataFork().seek(transferRequest.local.getDataFork().size());
                
                /* Receive the data fork. */
                
                try {
                    
                    transferrer.transferDataFork(xfinput, transferRequest.local.getDataFork().getDataOutput(), size);
                    
                } finally {
                    
                    transferRequest.local.setDataSize(transferRequest.local.getDataSize() + transferrer.getDataBytesWritten());
                    transferRequest.local.getDataFork().close();
                    
                }
                
                if(fti.numberOfBlocks == 3) {
                    
                    DebuggerOutput.debug("TransferThread[" + fth.ref + "]: " + "Expecting resource fork ...");
                    
                    xfinput.skipBytes(8);
                    xfinput.readFully(l, 0, 8);
                    size = ToArrayConverters.byteArrayToLong(l);
                    
                    transferRequest.local.getResourceFork().seek(transferRequest.local.getResourceFork().size());
                    
                    DebuggerOutput.debug("TransferThread[" + fth.ref + "]: " + "Receiving resource fork (" + size + " bytes).");
                    
                    /* Receive the resource fork. */
                    
                    try {
                        
                        transferrer.transferResourceFork(xfinput, transferRequest.local.getResourceFork().getDataOutput(), size);
                        
                    } finally {
                        
                        transferRequest.local.setResourceSize(transferRequest.local.getResourceSize() + transferrer.getResourceBytesWritten());
                        transferRequest.local.getResourceFork().close();
                        
                    }
                    
                    MacFileUtils.setFileTypeAndCreator(new File(transferRequest.local.getFile().getParent(), FilenameUtils.qualify(transferRequest.local.getFile().getName())), fti.fileType.getBytes(), fti.fileCreator.getBytes());
                    
                }
                
                /* Successfull transfer, so get rid of .hpf extension. */
                
                DebuggerOutput.debug("TransferThread[" + fth.ref + "]: " + "Receive OK, renaming file ...");
                transferRequest.local.renameTo(new File(transferRequest.local.getFile().toString().substring(0, transferRequest.local.getFile().toString().length() - 4)));
                
                xfoutput.flush();
                xfoutput.close();
                xfinput.close();
                
                stopMeter(fth.ref);
                
            } else {
                
                /* Prepare to send a file. */
                
                client.setSendBufferSize(HLProtocol.SOCK_SEND_BUF_SIZE);
                
                DebuggerOutput.debug("TransferThread[" + fth.ref + "]: " + "Sending a file ...");
                
                long dataSize = ((Long) transferRequest.rflt.chunks.get
                                 ("DATA")).longValue();
                long rsrcSize = ((Long) transferRequest.rflt.chunks.get
                                 ("MACR")).longValue();
                long dataRemaining = transferRequest.local.
                    getDataFork().size() - dataSize;
                long rsrcRemaining = transferRequest.local.
                    getResourceFork().size() - rsrcSize;
                
                MacFile hostPath = transferRequest.local;
                
                if(transferRequest.forkOrFile != HLProtocol.DL_DATA_FORK) {
                    
                    /* Don't send the header if the client only
                       wants the data fork (view file). */
                    
                    transferServer.hls.hlp.new FileTransferInfo
                        (hostPath.getFile().getName(),
                         hostPath.getType(),
                         hostPath.getCreator(),
                         hostPath.getComment(),
                         hostPath.getCreationDate(),
                         hostPath.getModificationDate(),
                         hostPath.getFinderFlags()).write(xfoutput);
                    
                }
                
                /* Create a transferrer to handle the actual
                   transfer and inform us about progress. */
                
                Transferrer transferrer = new Transferrer(0, fth.ref, this);
                transferrer.setDataInputStream(xfinput);
                transferrer.setDataOutputStream(xfoutput);
                
                /* Start the meter before we throw exceptions, so
                   they show up in the logfile (through
                   stopMeterWithError). */
                
                startMeter(transferrer, fth.ref, transferRequest.local.getFile().toString(), (transferRequest.forkOrFile == HLProtocol.DL_DATA_FORK ? dataRemaining : dataRemaining + rsrcRemaining));
                
                if(dataRemaining < 0 || rsrcRemaining < 0) 
                    throw new IOException("Attempt to resume beyond end of file.");
                
                /* Send the data fork. */
                
                if(transferRequest.forkOrFile != HLProtocol.DL_DATA_FORK) {
                    
                    xfoutput.write(new String("DATA").getBytes());
                    xfoutput.writeInt(0);
                    xfoutput.writeLong(dataRemaining);
                    
                }
                
                if(dataRemaining > 0) {
                    
                    DebuggerOutput.debug("Sending data fork, " + dataRemaining + " bytes.");
                    
                    transferRequest.local.getDataFork().seek(dataSize);
                    transferrer.transferDataFork(transferRequest.local.getDataFork().getDataInput(), xfoutput, dataRemaining);                
                    
                }
                
                /* Send the resource fork. */
                
                xfoutput.write(new String("MACR").getBytes());
                xfoutput.writeInt(0);
                xfoutput.writeLong(rsrcRemaining);
                
                if(rsrcRemaining > 0 &&
                   transferRequest.forkOrFile != HLProtocol.DL_DATA_FORK) {
                    
                    DebuggerOutput.debug("Sending resource fork, " + rsrcRemaining + " bytes.");
                    
                    transferRequest.local.getResourceFork().seek(rsrcSize);
                    transferrer.transferResourceFork(transferRequest.local.getResourceFork().getDataInput(), xfoutput, rsrcRemaining);
                    
                }
                
                if(!HLProtocol.asynchronousFileTransfers) {
                    
                    try {
                        
                        /* Wait for the client to close the socket. */
                        
                        DebuggerOutput.debug("Waiting for client to close socket ...");
                        xfinput.read();
                        
                    } catch(IOException e) {}
                    
                }

                DebuggerOutput.debug("Closing socket.");
                
                xfoutput.flush();
                xfoutput.close();
                xfinput.close();
                
                stopMeter(fth.ref);
                
            }
            
        } catch(InterruptedException e) {
            
        } catch(IOException e) {

            DebuggerOutput.stackTrace(e);
            if(file != null)
                stopMeterWithError(transferRequest.ref, e);

        } finally {

            killThread.disconnect();

        }

    }

    synchronized void standInLine() throws InterruptedException, IOException {

        while(transferRequest.queuePosition > 0) {
                    
            /* Stand in line if we're queued. */
            
            wait(4000);

            try {

                if(client.getInputStream() == null)
                    throw new IOException("standInLine tickle failed");

            } catch(NullPointerException e) {

                throw new IOException("standInLine tickle failed");

            }

            if(transferRequest.cancelled)
                throw new InterruptedException("stand in line cancelled");

        }

    }

    synchronized void nextInLine() {

        notify();

    }

    public void interrupt() {

        try {

            if(client != null)
                client.close();

        } catch(IOException e) {}
        
        super.interrupt();

    }

    public String toString() {

        String s = "TransferThread[";

        if(client != null) 
            s += "address: " + client.getInetAddress().toString();
        else
            s += "address: " + null;
       
        s += ", transferRequest: " + transferRequest;

        s += "]";

        return s;

    }

    /**
     * Following methods implement Meter.
     */

    String file = null;
    long progress = 0, total = 0, startTime = 0;
    MeterSource meterSource;
    
    public void startMeter(MeterSource ms, int key, String s, long size) {
     
        meterSource = ms;
        file = s;
        startTime = System.currentTimeMillis();
        total = size;

        transferRequest.totalSize = size;
        transferServer.hls.log(client.getInetAddress(), (transferRequest.type == HLServer.TransferRequest.FILE_UPLOAD ? "Receiving file " : "Sending file ") + s + " (" + size + " bytes).");

    }

    public void progressMeter(int key, long done) {

        transferRequest.progressDone = done;
        progress = done;

    }

    public void stopMeter(int key) {
        
        long now = System.currentTimeMillis();
        String speed = (now - startTime) / 1000 == 0 ? 
            "immeasurably fast" : 
            BytesFormat.format(total / ((now - startTime) / 1000)) + "/s";
        String size = total + " bytes" + (total < 1024 ? "" : " (" + BytesFormat.format(total) + ")");

        transferServer.hls.log(client.getInetAddress(), (transferRequest.type == HLServer.TransferRequest.FILE_UPLOAD ? "Received " : "Sent ") + file + ", total of " + size + " in " + TimeFormat.format((now - startTime) / 1000) + " (" + speed + ").");

    }
    public void stopMeterWithError(int key, Throwable t) {

        if(t instanceof Exception)
            DebuggerOutput.stackTrace((Exception) t);

        transferServer.hls.log(client.getInetAddress(), "Error " + (transferRequest.type == HLServer.TransferRequest.FILE_UPLOAD ? "receiving" : "sending") + " file " + file + " (" + t.toString() + ").");
        
    }
    public MeterSource getMeterSource() {

        return meterSource;

    }

}
