/**
 * $Id: TransferInterface.java,v 1.14 2001/10/09 02:02:03 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.Vector;
import java.util.Hashtable;

import javax.swing.*;

import redlight.hotline.*;
import redlight.utils.*;
import redlight.script.*;
import redlight.macfiles.MacFile;

/**
 * This class implements code common to uploading and downloading
 * of files.
 */
public abstract class TransferInterface 
    extends HLClientAdapter
    implements Meter, 
	       Queueable, 
	       Child, 
	       Scriptable {
    Machine rlm;
    JLabel info;
    JProgressBar progressBar;
    Container contentPane;
    MeterSource transferrer;	
    boolean interrupted = false, transferStarted = false, taskComplete = false;
    int retryCount = 1;
    int maxRetries = 100;
    int transferTask = -1;
    int xfref = 0;
    long total;
    long startTimeMillis = 0, stallTime = 0;
    long previousProgress = 0;
    String filename;
    MacFile localFile = null;
    boolean closeUnregistered = false;
    Meter meter;
    boolean closeLocalFile = true;
    boolean autoResume = false, autoRetry = false;
    String path;
    HLProtocol.FileListComponent file;
    int progressSampleCounter = 0;

    public TransferInterface(Machine rlm, 
                             String filename) {

	this.filename = filename;
	this.rlm = rlm;
        this.meter = this;
	this.maxRetries = 
            Main.rlo.getIntegerProperty("Auto.DownloadRetries");
        Main.activeTransfers++;
        rlm.getInterface().getShell().activeTransfers++;
	rlm.getInterface().registerChild(this);
	rlm.getScriptBroker().addTarget(this);

        /* Get a content pane to draw ourselves in. */

        try {
            
            ScriptResult[] sr = rlm.getScriptBroker().
                executeScript(new ScriptObject(Scripting.SWALLOW_REQUEST, 
                                               this));
            contentPane = (Container) sr[0].getUserObject();
            
        } catch (UnknownMessageException e) {
            
            e.printStackTrace();
            
        }

	contentPane.setBackground(rlm.getSchemeColor("background"));
	contentPane.setForeground(rlm.getSchemeColor("foreground"));
	contentPane.setLayout(new BorderLayout());
	info = new JLabel(filename + "   [waiting for server to respond]        ");
	info.setFont((Font) Main.rlo.getProperty("Font.list"));
	info.setBackground(rlm.getSchemeColor("background"));
	info.setForeground(rlm.getSchemeColor("foreground"));

        if(Main.iconFile != null)
            info.setIcon(Main.iconFile.getIconOrPlaceholder(400));

	contentPane.add(info, BorderLayout.SOUTH);
	progressBar = new JProgressBar(0, 0);
	contentPane.add(progressBar, BorderLayout.CENTER);

    }

    public File getLocalFile() {
        /*
        try {

            DebuggerOutput.debug("TransferInterface.getLocalFile[" + transferTask + "]: waiting for completion ...");
            while(!closeUnregistered) 
                wait();
            DebuggerOutput.debug("TransferInterface.getLocalFile[" + transferTask + "]: got completion.");
        */  
            if(localFile == null)
                return null;
            
            DebuggerOutput.debug("TransferInterface.getLocalFile[" + transferTask + "]: returning " + localFile.getFile().toString());
            return localFile.getFile();
            /*
            
        } catch(InterruptedException e) {
            
            return null;

        }
            */

    }

    /**
     * This function is called by the subclasss after it's initialization. 
     * If the request can be granted (doesn't have to be queued),
     * then granted() is invoked on the subclass, which should 
     * perform the actual action for downloading or uploading.
     */
    protected void doRequest() {

	if(Main.rlo.getBooleanProperty("Toggle.QueueDownloads") && 
           (this instanceof DownloadInterface ||
            this instanceof ViewInterface ||
            this instanceof FolderDownloadInterface)) {

            if(rlm.addTransfer(this)) {
                
                SwingUtilities.invokeLater(new Runnable() {
                        
                        public void run() {
                            
                            info.setText(filename + 
                                         (retryCount > 1 ? 
                                          "   [queued, retry " + 
                                          retryCount + "]" :
                                          "   [queued]   "));
                            repaintStatus();
                            
                        }
                        
                    });
                
            }
            
	} else {
	                                                        
            
            SwingUtilities.invokeLater(new Runnable() {
                    
                    public void run() {
                        
                        info.setText(filename + "   [waiting for server to respond]     ");
                        repaintStatus();
                        
                    }
                    
                });
            
	    granted();
	    
	}

    }
    
    /**
     * This function is called by the subclass after it has been
     * granted a transfer (see doRequest). This function installs 
     * the class as the error handler for the task number of the 
     * transfer task. This causes errors on the transfer task
     * to be routed (by the Machine) to us, not to the central error
     * handler. This way we can implement retry functionality.
     */
    void setTransferTask(int t) {

	transferTask = t;
	rlm.addTaskListener(transferTask, this);

    }

    public void retry(final String error) {
        
        startTimeMillis = 0;

        rlm.removeTransfer(this);

        interrupt();

        /* Try again if we haven't exhausted our number of retries,
           otherwise show the error from the server and die. */
        
	if(retryCount >= maxRetries) {

            rlm.getInterface().error(filename + ": " + error);
            localFile = null;            
            close();
            return;

        }
        
        /* Not really interrupted :) */

        interrupted = false;
        autoResume = true;

        SwingUtilities.invokeLater(new Runnable() {
                
                public void run() {
                    
                    info.setText(filename + "   [try " + retryCount++ + ": " + error + "]");
                    repaintStatus();
                    
                }
                
            });
        
        new Thread(new Runnable() {

                public synchronized void run(){ 

                    try {
                        
                        /* Wait a bit before trying again. */
                    
                        DebuggerOutput.debug("RetryThread[" + transferTask + "]: waiting for retry...");

                        do {

                            wait(5000);

                        } while(!taskComplete && !interrupted);

                        DebuggerOutput.debug("RetryThread[" + transferTask + "]: woke up from retry wait");
                    
                        taskComplete = false;

                        if(!interrupted) {
                        
                            DebuggerOutput.debug("RetryThread[" + transferTask + "]: retrying ...");
                            doRequest();
                        
                        }
                    
                    } catch(InterruptedException e) {
                    
                        DebuggerOutput.debug("RetryThread[" + transferTask + "]: interrupted waiting for retry");
                    
                    }
                
                    DebuggerOutput.debug("RetryThread[" + transferTask + "]: exiting");
                
                }

            }, "RetryThread[" + transferTask + "]").start();
        
        
    }

    protected void interrupt() {

	interrupted = true;

	if(transferrer != null) {

            DebuggerOutput.debug("TransferInterface.interrupt: interrupting transferrer ...");
	    transferrer.interrupt();

        }

    }
 
    /**
     * Following functions implement Queueable.
     */
    public abstract void granted();

    /**
     * Following functions implement Child.
     */
    public void close() {

        try {
            
            if(closeLocalFile) 
                if(localFile != null)
                    localFile.close();
            
        } catch (IOException ex) {}
        
        rlm.getInterface().unregisterChild(this);

	interrupt();

        try {
            
            rlm.getScriptBroker().
                executeScript(new ScriptObject(Scripting.UNSWALLOW_REQUEST,
                                               contentPane));
        } catch (UnknownMessageException e) {}
        
        rlm.removeTransfer(this);
        rlm.getInterface().getShell().activeTransfers--;
        Main.activeTransfers--;
        
	rlm.getScriptBroker().removeTarget(this);

    }

    public void displayPropertyChanged(String what, Object property) {

        SwingUtilities.invokeLater(new Runnable() {

                public void run() {

                    contentPane.setBackground(rlm.getSchemeColor("background"));
                    contentPane.setForeground(rlm.getSchemeColor("foreground"));
                    info.setBackground(rlm.getSchemeColor("background"));
                    info.setForeground(rlm.getSchemeColor("foreground"));
                    contentPane.repaint();

                }

            });

    }

    /**
     * Following functions implement Meter.
     */
    public void startMeter(MeterSource ms, int ref, String f, final long size) {
        transferrer = ms;
	transferStarted = true;
	total = size;
        xfref = ref;
        DebuggerOutput.debug("TransferInterface.startMeter[" + transferTask + "]: total = " + total);

        SwingUtilities.invokeLater(new Runnable() {
                            
                public void run() {

                    info.setText(filename + " (0 / "+BytesFormat.format(size)+")");
                    repaintStatus();

                }

            });

	progressBar.setMaximum((int) (size / 10));

        /*
	if(swallow) {

            try {
                
                rlm.getScriptBroker().
                    executeScript(Scripting.REPAINT_MESSAGE);
                
            } catch (UnknownMessageException e) {}
            
	}
        */

    }

    boolean isStalled(long progress) {

        if(previousProgress == 0) {

            /* First time. */

            previousProgress = progress;
            return false;

        }

        if(progress - previousProgress > 0) {

            /* Not stalled. */

            stallTime = 0;
            return false;

        }
                
        if(stallTime == 0) {

            /* Stalled for the first time. */

            stallTime = System.currentTimeMillis();
            return false;

        }

        /* Close and retry if stalled for too long. */

        int maxStallSeconds =
            Main.rlo.getIntegerProperty("Auto.StallTimeout");

        if(maxStallSeconds > 0) {

            if(System.currentTimeMillis() - stallTime > 
               (maxStallSeconds * 1000)) {

                stallTime = 0;
                previousProgress = 0;
                retry("Transfer stalled.");
                return true;

            }
            
        }

        return false;

    }

    public void progressMeter(int ref, final long progress) {

	if(startTimeMillis == 0)
            startTimeMillis = System.currentTimeMillis();
        
        if(isStalled(progress))
            return;

	long time_d = System.currentTimeMillis() - startTimeMillis;
	long bytes_d = progress;
	final long bytes_sec = bytes_d / ((time_d / 1000) + 1);
        final long secs_left = (total - progress) / 
            (bytes_sec == 0 ? 1 : bytes_sec);

        SwingUtilities.invokeLater(new Runnable() {

                public void run() {

                    info.setText(filename + " (" + 
                                 BytesFormat.format(progress) + " / " +
                                 BytesFormat.format(total) + " " +
                                 BytesFormat.format(bytes_sec) + "/s ETA: " +
                                 TimeFormat.format(secs_left) + ")");
                    progressBar.setValue((int) (progress / 10));
                    repaintStatus();

                }

            });

	previousProgress = progress;

    }

    public void stopMeter(int ref) {

        SwingUtilities.invokeLater(new Runnable() {

                public void run() {

                    info.setText(filename + " 100% (" + BytesFormat.format(total) + ")");
                    repaintStatus();

                }

            });

	rlm.playAudio("filedone");

	close();

    }

    public void stopMeterWithError(int ref, Throwable t) {

        DebuggerOutput.debug("TransferInterface.stopMeterWithError[" + transferTask + "]: error = " + t.toString());
        
        String error = t.toString();

	if(t instanceof Exception && t.getMessage() != null) 
            error = t.getMessage();
        
        retry(error);

    }
    
    public MeterSource getMeterSource() {

	return transferrer;

    }
	
    /**
     * Following functions extend HLClientAdapter.
     */
    public void handleTransferQueuePosition(int ref, final int position) {

        DebuggerOutput.debug("TransferInterface.handleTransferQueuePosition[" + transferTask + "]: ref = " + ref + ", xfref = " + xfref);

        if(ref == xfref) {

            SwingUtilities.invokeLater(new Runnable() {
                    
                    public void run() {

                        info.setText(filename + "   [remotely queued #" + position + "]   ");
                        repaintStatus();

                    }

                });

        }

    }

    public void handleTaskComplete(int task) {

        DebuggerOutput.debug("TransferInterface.handleTaskComplete[" + transferTask + "]: task " + task + " complete.");

        taskComplete = true;

    }

    public void handleTaskError(int task, String error) {

        DebuggerOutput.debug("TransferInterface.handleTaskError[" + transferTask + "]: task " + task + " error: " + error);

        taskComplete = true;
        retry(error);

    }

    /**
     * Following functions implement Scriptable.
     */
    public void gotTarget(ScriptBroker sb) {}
    public void lostTarget(ScriptBroker sb) {}
    public long getKnownMessages() {

	return Scripting.CLOSE_WINDOW;

    }

    public ScriptResult executeScript(ScriptObject s) 
	throws UnknownMessageException {

	int value = (int) s.getType();   // losing 32 bits of precision here

	switch(value) {

	case Scripting.CLOSE_WINDOW:

	    if((Container) s.getUserObject() == contentPane) {

		close();
		return new ScriptResult(this, null);

	    }

	}

	throw new UnknownMessageException("Unknown message");

    }

    public void repaintStatus() {

        info.repaint();

        try {
            
            rlm.getScriptBroker().
                executeScript(Scripting.REPAINT_MESSAGE);
                
        } catch (UnknownMessageException e) { }

    }

}
