/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui.table;

import java.awt.Color;
import java.awt.Component;

import javax.swing.JCheckBox;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.table.TableCellRenderer;

/**
 * The Class BooleanRenderer.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class BooleanRenderer extends JCheckBox implements TableCellRenderer {
  /** The supports focus border. */
  protected boolean           supportsFocusBorder;
  
  /** The tooltip enabled. */
  protected boolean           tooltipEnabled;
  
  /** The alternate row color enabled. */
  protected boolean           alternateRowColorEnabled = true;
  
  /** The alternate row color. */
  protected Color             alternateRowColor;

  /** The Constant DEFAULT_NO_FOCUS_BORDER. */
  private static final Border DEFAULT_NO_FOCUS_BORDER  = new EmptyBorder(1, 1, 1, 1);

  /** UID. */
  private static final long   serialVersionUID         = 1569030349242768678L;

  /**
   * Instantiates a new boolean renderer.
   */
  public BooleanRenderer() {
    this(CENTER, true);
  }

  /**
   * Instantiates a new boolean renderer.
   *
   * @param alignment the alignment
   */
  public BooleanRenderer(int alignment) {
    this(alignment, true);
  }

  /**
   * Instantiates a new boolean renderer.
   *
   * @param alignment the alignment
   * @param tooltipEnabled the tooltip enabled
   */
  public BooleanRenderer(int alignment, boolean tooltipEnabled) {
    this.tooltipEnabled = tooltipEnabled;
    setOpaque(true);
    setHorizontalAlignment(alignment);
    //setBorderPainted(true);
    setSupportsFocusBorder(true);
  }

  /**
   * Sets the supports focus border.
   *
   * @param supportsFocusBorder the new supports focus border
   */
  public void setSupportsFocusBorder(boolean supportsFocusBorder) {
    this.supportsFocusBorder = supportsFocusBorder;
    if (supportsFocusBorder) {
      setBorder(getNoFocusBorder());
    }
  }

  /**
   * Gets the supports focus border.
   *
   * @return the supports focus border
   */
  public boolean getSupportsFocusBorder() {
    return supportsFocusBorder;
  }

  /**
   * Gets the no focus border.
   *
   * @return the no focus border
   */
  protected static Border getNoFocusBorder() {
    Border border = UIManager.getBorder("Table.cellNoFocusBorder"); //$NON-NLS-1$
    if (border != null)
      return border;
    return DEFAULT_NO_FOCUS_BORDER;
  }

  /**
   * Checks if is tooltip enabled.
   *
   * @return true, if is tooltip enabled
   */
  public boolean isTooltipEnabled() {
    return tooltipEnabled;
  }

  /**
   * Sets the tooltip enabled.
   *
   * @param tooltipEnabled the new tooltip enabled
   */
  public void setTooltipEnabled(boolean tooltipEnabled) {
    this.tooltipEnabled = tooltipEnabled;
  }

  /**
   * Checks if is alternate row color enabled.
   *
   * @return true, if is alternate row color enabled
   */
  public boolean isAlternateRowColorEnabled() {
    return alternateRowColorEnabled;
  }

  /**
   * Sets the alternate row color enabled.
   *
   * @param alternateRowColorEnabled the new alternate row color enabled
   */
  public void setAlternateRowColorEnabled(boolean alternateRowColorEnabled) {
    this.alternateRowColorEnabled = alternateRowColorEnabled;
  }

  /**
   * Gets the alternate row color.
   *
   * @return the alternate row color
   */
  public Color getAlternateRowColor() {
    return alternateRowColor;
  }

  /**
   * Sets the alternate row color.
   *
   * @param alternateRowColor the new alternate row color
   */
  public void setAlternateRowColor(Color alternateRowColor) {
    this.alternateRowColor = alternateRowColor;
    this.alternateRowColorEnabled = alternateRowColor != null;
  }

  /**
   * Sets the colors.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus the has focus
   * @param row the row
   * @param column the column
   */
  protected void setColors(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
    TableUtilities.setColors(this, table, isSelected, row, alternateRowColorEnabled, alternateRowColor);
  }

  /**
   * Sets the border.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus the has focus
   * @param row the row
   * @param column the column
   */
  protected void setBorder(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
    TableUtilities.setBorder(this, table, isSelected, hasFocus, row, column, getNoFocusBorder());
  }

  /**
   * Sets the state.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus the has focus
   * @param row the row
   * @param column the column
   */
  protected void setState(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
    //empty  
  }

  /**
   * Sets the value.
   *
   * @param table the table
   * @param value the value
   * @param row the row
   * @param column the column
   */
  protected void setValue(JTable table, Object value, int row, int column) {
    if (value != null) {
      if (value instanceof Boolean)
        setSelected((Boolean) value);
      else
        setSelected(Boolean.valueOf(value.toString()));
    }
  }

  /**
   * Sets the font.
   *
   * @param table the table
   * @param value the value
   * @param row the row
   * @param column the column
   */
  protected void setFont(JTable table, Object value, int row, int column) {
    //setFont(table.getFont());
  }

  /**
   * Gets the table cell renderer component.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus has focus
   * @param row the row
   * @param column the column
   * @return the table cell renderer component
   * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
   */
  @Override
  public Component getTableCellRendererComponent(JTable table,
                                                 Object value,
                                                 boolean isSelected,
                                                 boolean hasFocus,
                                                 int row,
                                                 int column) {

    setColors(table, value, isSelected, hasFocus, row, column);

    setFont(table, value, row, column);

    if (supportsFocusBorder) {
      setBorder(table, value, isSelected, hasFocus, row, column);
    }

    setState(table, value, isSelected, hasFocus, row, column);

    setValue(table, value, row, column);

    return this;
  }

}
