/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui.table;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.logging.Level;
import java.util.logging.Logger;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Font;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.table.JTableHeader;

import jnpad.ui.JNPadLabel;
import jnpad.ui.layout.VerticalFlowLayout;
import jnpad.ui.plaf.LAFUtils;
import jnpad.util.Utilities;

/**
 * The Class MultiLineHeaderTableCellRenderer.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class MultiLineHeaderTableCellRenderer extends JPanel implements HeaderTableCellRenderer {
  /** The Constant TOP. */
  public static final int     TOP              = VerticalFlowLayout.TOP;

  /** The Constant MIDDLE. */
  public static final int     MIDDLE           = VerticalFlowLayout.MIDDLE;

  /** The Constant BOTTOM. */
  public static final int     BOTTOM           = VerticalFlowLayout.BOTTOM;

  private JPanel              panel            = new JPanel();
  private JLabel              label            = new JNPadLabel();

  /** Logger */
  private final static Logger LOGGER           = Logger.getLogger(MultiLineHeaderTableCellRenderer.class.getName());

  /** UID */
  private static final long   serialVersionUID = -6433259089696708378L;

  /**
   * Instantiates a new multi line header table cell renderer.
   *
   * @param align int
   */
  public MultiLineHeaderTableCellRenderer(int align) {
    super(new BorderLayout());
    add(panel, BorderLayout.CENTER);
    add(label, BorderLayout.EAST);
    panel.setLayout(new VerticalFlowLayout(align, 0, 0));
    panel.setOpaque(true);
    label.setOpaque(true);
    label.setHorizontalAlignment(SwingConstants.CENTER);
  }

  /**
   * Gets the table cell renderer component.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus has focus
   * @param row the row
   * @param column the column
   * @return the table cell renderer component
   * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
   */
  @Override
  public Component getTableCellRendererComponent(JTable table,
                                                 Object value,
                                                 boolean isSelected,
                                                 boolean hasFocus,
                                                 int row,
                                                 int column) {
    panel.removeAll();

    int sortedColumnIndex = -1;
    boolean isSortedColumnAscending = true;

    if (table instanceof JNPadTable) {
      sortedColumnIndex = ( (JNPadTable) table).getSortedColumnIndex();
      isSortedColumnAscending = ( (JNPadTable) table).isSortedColumnAscending();
    }

    Color foreground = LAFUtils.getTableHeaderForeground();
    Color background = LAFUtils.getTableHeaderBackground();
    Font font = LAFUtils.getTableHeaderFont();

    if (table != null) {
      JTableHeader header = table.getTableHeader();
      if (header != null) {
        label.setForeground(foreground = header.getForeground());
        label.setBackground(background = header.getBackground());
        label.setFont(font = header.getFont());
        panel.setForeground(foreground);
        panel.setBackground(background);
        panel.setFont(font);
      }
    }

    label.setIcon(column == sortedColumnIndex ?
                  (isSortedColumnAscending ? DECENDING : ASCENDING) : NONSORTED);

    String str = Utilities.toString(value);
    BufferedReader br = new BufferedReader(new StringReader(str));
    String line;
    try {
      while ((line = br.readLine()) != null) {
        JLabel l = new JNPadLabel();
        l.setText(line);
        l.setHorizontalAlignment(SwingConstants.CENTER);
        l.setOpaque(false);
        l.setForeground(foreground);
        l.setFont(font);
        panel.add(l);
      }
    }
    catch (IOException ex) {
      LOGGER.log(Level.WARNING, ex.getMessage(), ex);
    }

    panel.validate();

    setBorder(LAFUtils.getTableHeaderCellBorder());

    return this;
  }
}
