/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text.highlighter;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;

import javax.swing.plaf.TextUI;
import javax.swing.text.BadLocationException;
import javax.swing.text.JTextComponent;

import jnpad.config.Config;

/**
 * The Class OccurrenceHighlightPainter.
 * 
 * @version 0.3
 * @since jNPad 0.1
 * @see IHighlightPainter
 */
public class OccurrenceHighlightPainter implements IHighlightPainter {
  private Color _color;
  private Color _borderColor;

  /**
   * Instantiates a new occurrence highlight painter.
   */
  public OccurrenceHighlightPainter() {
    configure(CFG_COLOR);
  }

  /**
   * Configure.
   *
   * @param cfg the cfg
   * @see jnpad.config.Configurable#configure(int)
   */
  @Override
  public void configure(final int cfg) {
    if ((cfg & CFG_COLOR) != 0) {
      setColor(Config.OCCURRENCES_HIGHLIGHTER_COLOR.getValue());
      setBorderColor(Config.OCCURRENCES_HIGHLIGHTER_BORDER_COLOR.getValue());
    }
  }

  /**
   * Sets the color.
   *
   * @param color the new color
   */
  public void setColor(Color color) {
    if (color != null) {
      _color = color;
    }
  }

  /**
   * Gets the color.
   *
   * @return the color
   */
  public Color getColor() {
    return _color;
  }

  /**
   * Gets the border color.
   *
   * @return the border color
   */
  public Color getBorderColor() {
    return _borderColor;
  }

  /**
   * Sets the border color.
   *
   * @param color the new border color
   */
  public void setBorderColor(Color color) {
    if (color != null) {
      _borderColor = color;
    }
  }

  /**
   * Paint.
   *
   * @param g the Graphics
   * @param offs0 the offs0
   * @param offs1 the offs1
   * @param bounds the bounds
   * @param c the JTextComponent
   * @see javax.swing.text.Highlighter.HighlightPainter#paint(java.awt.Graphics, int, int, java.awt.Shape, javax.swing.text.JTextComponent)
   */
  @Override
  public void paint(Graphics g, int offs0, int offs1, Shape bounds, JTextComponent c) {
    final Rectangle alloc = bounds.getBounds();
    try {
      // --- determine locations ---
      TextUI mapper = c.getUI();
      Rectangle r0 = mapper.modelToView(c, offs0);
      Rectangle r1 = mapper.modelToView(c, offs1);

      // --- render ---
      g.setColor(getColor());
      if (r0.y == r1.y) {
        Graphics2D g2D = (Graphics2D) g;
        final Object oldRendering    = g2D.getRenderingHint(RenderingHints.KEY_RENDERING);
        final Object oldAntialiasing = g2D.getRenderingHint(RenderingHints.KEY_ANTIALIASING);
        g2D.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
        g2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

        // same line, render a rectangle
        Rectangle r = r0.union(r1);
        g.fillRoundRect(r.x, r.y, r.width, r.height, 4, 4);
        g.setColor(getBorderColor());
        g.drawRoundRect(r.x, r.y, r.width - 1, r.height - 1, 4, 4);

        g2D.setRenderingHint(RenderingHints.KEY_RENDERING, oldRendering);
        g2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, oldAntialiasing);
      }
      else {
        // different lines
        int p0ToMarginWidth = alloc.x + alloc.width - r0.x;
        g.fillRect(r0.x, r0.y, p0ToMarginWidth, r0.height);
        if ((r0.y + r0.height) != r1.y) {
          g.fillRect(alloc.x, r0.y + r0.height, alloc.width, r1.y - (r0.y + r0.height));
        }
        g.fillRect(alloc.x, r1.y, (r1.x - alloc.x), r1.height);
      }
    }
    catch (BadLocationException e) {
      // can't render
    }
  }

}
