package de.beagtex.gui;

import java.awt.Font;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.StringTokenizer;

import javax.swing.JOptionPane;
import javax.swing.JTextArea;
import javax.swing.undo.UndoManager;

import de.beagtex.control.RedoAction;
import de.beagtex.control.UndoAction;
import de.beagtex.listener.control.MyKeyListener;
import de.beagtex.listener.control.MyMouseListener;
import de.beagtex.listener.control.MyUndoableEditListener;

/**
 * An Editor is an extended JTextArea. It supports undo//redo/operations,
 * search/replace and opening/saving of Tex-Files
 * 
 * @author Benjamin Haag
 * 
 */
public class Editor extends JTextArea {

	/**
	 * serialVersionUID
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * UndoManager
	 */
	protected UndoManager undoMan = new UndoManager();

	/**
	 * UndoAction
	 */
	public UndoAction undoAction = new UndoAction(this);

	/**
	 * RedoAction
	 */
	public RedoAction redoAction = new RedoAction(this);

	/**
	 * related BFrame which the editor is embedded in
	 */
	private BFrame bFrame;

	/**
	 * File-variable save-options
	 */
	private File savefile;

	/**
	 * Shows if a File is already saved. Standard is false, will be set to true
	 * during save-operation. every change sets it back to false.
	 * 
	 */
	private boolean saved = false;

	/**
	 * Number of opened Editor-Windows; is incremented/decremented while
	 * open/close. When the last windows is closed (number = 0) also the JVM
	 * should be closed.
	 */
	private static int number = 1;

	/**
	 * Constructor: an editor-element is always related to a BFrame, that has to
	 * be passed to it here. Furthermore Key-, Mouse- und UndoableEditListener
	 * are added
	 * 
	 * @see MyKeyListener
	 * @see MyMouseListener
	 * @see MyUndoableEditListener
	 * 
	 * @param pBFrame :
	 *            BFrame which the Editor is related to
	 */
	public Editor(BFrame pBFrame) {

		this.bFrame = pBFrame;
		setLineWrap(true);
		setWrapStyleWord(true);
		this.addKeyListener(new MyKeyListener(this));
		Font t = new Font("eigen", Font.ROMAN_BASELINE, 16);
		setFont(t);
		this.addMouseListener(new MyMouseListener(this));
		// for Connection with the Undo-Manager
		getDocument().addUndoableEditListener(new MyUndoableEditListener(this));
		// loadProperties(BEAGTEX_DEFAULT_PROP);
		// loadProperties("data/text_ger.properties");
	}

	/**
	 * Opens a File via FileDialog; fills variable "savefile"; Calls:
	 * OpenListener (connected to StartChoice, Open-Button/-Menu
	 * 
	 */
	public void open() {
		try {
			MyOpenChooser oc = new MyOpenChooser(this);
			oc.setVisible(true);

			if (oc.getSelectedFile() != null) {
				String choice = oc.getSelectedFile().getName();
				getBFrame().setTitle("BeaGTex - " + choice);
				FileReader fily = new FileReader(oc.getSelectedFile());
				BufferedReader buffyRead = new BufferedReader(fily);
				savefile = oc.getSelectedFile();
				boolean eof = false;
				while (!eof) {
					String line = buffyRead.readLine();
					if (line == null) {
						eof = true;
						line = "";
					} else {
						this.getDocument();
						this.append(line + "\r\n");
					}
				}
			} else
				return;

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Saves a file. Is the file new, saveAs ist atomatically called
	 */
	public void save() {

		try {
			if (savefile != null) {
				this.getBFrame().setTitle("BeaGTex - " + savefile);
				StringBuffer buffy = new StringBuffer();
				String text = this.getText();
				buffy.append(text);
				FileWriter fily = new FileWriter(savefile);
				fily.write(buffy.toString());
				fily.close();
				this.getBFrame().getBlButtons().getSaveB().setEnabled(false);
				this.setSaved(true);
				System.out
						.println("BeaGTex has saved your file to " + savefile);
				this.requestFocus();
				if (Editor.getNumber() == 0) {
					System.out.println("number: " + Editor.getNumber());
					System.exit(0);
				}
			} else
				saveAs();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Calls save-dialog; fills variable "savefile"
	 */
	public void saveAs() {

		try {
			MySaveChooser sc = new MySaveChooser(this);
			sc.setVisible(true);

			if (sc.getSelectedFile() != null) {
				String filename = sc.getSelectedFile().getName();
				this.getBFrame().setTitle("BeaGTex - " + filename);
				StringBuffer buffy = new StringBuffer();
				String text = this.getText();
				buffy.append(text);
				FileWriter fily = new FileWriter(sc.getSelectedFile());
				fily.write(buffy.toString());
				fily.close();
				this.getBFrame().getBlButtons().getSaveB().setEnabled(false);
				this.setSaved(true);
				savefile = sc.getSelectedFile();
				System.out
						.println("BeaGTex has saved your file to " + filename);
				this.requestFocus();
				if (Editor.getNumber() == 0) {
					System.out.println("number: " + Editor.getNumber());
					System.exit(0);
				}
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Method for searching; takes the whole text into a StringBuffer,
	 * determines the cursor-position and the end of the document. A second
	 * StringBuffer is build from cursor to doc-end wich a StringTokenizer is
	 * related to. Every token is compared to the searched term. For marking the
	 * indices of the StringBuffer are used.
	 * 
	 * @param searchWindow:
	 *            from the SearchWindow the searched String is identified
	 * @return found: Standard = false; when found --> true
	 */
	public boolean search(SearchWindow searchWindow) {

		boolean found = false;
		StringBuffer buffy = new StringBuffer(this.getText());
		int end = buffy.toString().length();
		int actual = this.getCaretPosition();

		StringBuffer searchBuffy;
		searchBuffy = new StringBuffer(buffy.substring(actual, end));

		StringTokenizer toky = new StringTokenizer(searchBuffy.toString(),
				" \t\n\r\f ! ? . , - ' ; : / { } \\ ");

		while (toky.hasMoreTokens()) {
			String test = toky.nextToken();
			String searchString = searchWindow.searchTF.getText();
			if (test.equalsIgnoreCase(searchString)) {
				int mStart = searchBuffy.toString().indexOf(searchString)
						+ actual;
				int mEnd = searchBuffy.toString().indexOf(searchString)
						+ searchString.length() + actual;
				this.select(mStart, mEnd);
				found = true;
			}
		}
		return found;
	}

	/**
	 * Replaces a searched String by another
	 * 
	 * @param searchWindow:
	 *            from the SearchWindow the alternative String is identiified
	 */
	public void replace(SearchWindow searchWindow) {

		String alternativeString = searchWindow.replaceTF.getText();
		if (this.getSelectedText() != null)
			this.replaceSelection(alternativeString);
		else
			JOptionPane.showMessageDialog(null, BFrame.get_prop().getProperty(
					"beagtex.editor.replace"), BFrame.get_prop().getProperty(
					"beagtex.editor.replacetitle"),
					JOptionPane.INFORMATION_MESSAGE);
	}

	/**
	 * Increments the Editor-Number-Variable
	 */
	public static void incrementNumber() {
		number++;
		System.out.println(number);
	}

	/**
	 * Decrements the Editor-Number-Variable
	 */
	public static void decrementNumber() {
		number--;
		System.err.println(number);
	}

	/**
	 * Resets the Undo-Managers
	 */
	protected void resetUndoManager() {
		undoMan.discardAllEdits();
		undoAction.update();
		redoAction.update();
	}

	// ab hier Getter und Setter
	public UndoManager getUndoMan() {
		return undoMan;
	}

	public BFrame getBFrame() {
		return bFrame;
	}

	public File getSaveFile() {
		return savefile;
	}

	public boolean isSaved() {
		return saved;
	}

	public void setSaved(boolean pSaved) {
		saved = pSaved;
	}

	public static int getNumber() {
		return number;
	}

	public static void setNumber(int pNumber) {
		number = pNumber;
	}

}
